package net.axay.memoire

import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.sync.Mutex

/**
 * This interface describes the functions a generic cache must have.
 * Please note that [net.axay.memoire.memory.MemoryCache] and
 * [net.axay.memoire.disk.DiskCache] might have additional functionality
 * not available in this base interface.
 *
 * @param KeyType the type of keys which values will be assigned to
 * @param ValueType the type of values which can be read from the cache
 * @param PutType the type of values which can be put into the cache
 */
abstract class Cache<KeyType, ValueType, PutType>(
    val validationConfig: CacheValidationConfig,
) {

    // lock resolving of new entries to avoid unnecessary calculations
    private val resolveMutex = Mutex()
    private val currentlyResolving = HashMap<KeyType, Deferred<ValueType?>>()

    /**
     * Returns the value associated with the given [key] or
     * null if the [key] is not present in the cache.
     */
    abstract suspend fun get(key: KeyType): ValueType?

    /**
     * Checks whether the given [key] is present in the cache.
     */
    abstract suspend fun contains(key: KeyType): Boolean

    /**
     * Stores the given [value] in this cache. It will be associated
     * to the given [key].
     * If the same key is already present in the cache and associated
     * to a different value, the value will be replaced.
     */
    abstract suspend fun put(key: KeyType, value: PutType)

    /**
     * Does the same as [put], but also reads the value from the cache in
     * the same operation. Especially useful for caches, where [PutType] and
     * [ValueType] aren't the same.
     */
    abstract suspend fun putAndGet(key: KeyType, value: PutType): ValueType?

    /**
     * Removes the value associated with the given [key], and then
     * returns true if the [key] was present in this cache.
     */
    abstract suspend fun remove(key: KeyType): Boolean

    /**
     * Removes the value associated with the given [key] from this cache
     * and returns this value, or null if there is no such value.
     */
    abstract suspend fun getAndRemove(key: KeyType): ValueType?

    /**
     * Removes the value associated with the given [key] from this cache
     * if it matches an invalidation criterion. This functions returns true
     * if a value was removed, false if the value was still valid and null
     * if no value was present.
     */
    abstract suspend fun removeIfInvalid(key: KeyType): Boolean?

    /**
     * Evicts all invalid entries in this cache. This means that all
     * entries will be checked against the [validationConfig]. Note that
     * this might take some time, especially for caches which operate
     * on disks.
     */
    abstract suspend fun evict()

    /**
     * Does the same as [getOrPut], but the returned value can also be null
     * if the result of [provider] is null (nothing will be put into the cache
     * in that case).
     *
     * This function also ensures that two [provider] calls won't happen at once.
     */
    suspend fun getOrPutOrNull(key: KeyType, provider: suspend () -> PutType?): ValueType? {
        get(key)?.let { return it }

        resolveMutex.lock()
        currentlyResolving[key]?.let {
            resolveMutex.unlock()
            return it.await()
        }

        val valueDeferred = CompletableDeferred<ValueType?>().also { currentlyResolving[key] = it }
        resolveMutex.unlock()

        val putValue = provider()
        if (putValue == null) {
            valueDeferred.complete(null)
            return null
        }

        val newValue = putAndGet(key, putValue)

        valueDeferred.complete(newValue)
        return newValue
    }
}

/**
 * Returns the value associated with the given [key].
 *
 * @throws IllegalStateException if the cache does not contain a value associated
 * with the [key]
 */
suspend fun <K, V> Cache<K, V, *>.getOrThrow(key: K): V {
    return get(key) ?: error("No value associated with the key '$key'")
}

/**
 * Does the same as [Cache.getAndRemove] but throws if the cache does not contain any
 * value associated with the given [key].
 */
suspend fun <K, V> Cache<K, V, *>.getOrThrowAndRemove(key: K): V {
    return getAndRemove(key) ?: error("No value associated with the given '$key'")
}

/**
 * Does the same as [Cache.getOrPutOrNull] but the result of [provider]
 * cannot be null, and therefore this function also does not return a nullable
 * value.
 */
suspend fun <K, V, P> Cache<K, V, P>.getOrPut(key: K, provider: suspend () -> P): V {
    return getOrPutOrNull(key, provider)!! // TODO use definitely non-null types
}
