package net.asynchorswim.ddd.view

import java.util.concurrent.atomic.AtomicLong

import com.typesafe.config.Config
import org.apache.lucene.analysis.standard.StandardAnalyzer
import org.apache.lucene.index.{DirectoryReader, IndexWriter, IndexWriterConfig, Term}
import org.apache.lucene.index.IndexWriterConfig.OpenMode
import org.apache.lucene.queryparser.classic.{QueryParser, QueryParserBase}
import org.apache.lucene.search.{IndexSearcher, TermQuery}
import org.apache.lucene.store.Directory
import resource._

import scala.util.Success

class LuceneView[A](directory: Directory, config: Config)(implicit mapper: LuceneViewMapper[A]) extends View[A, String, String] {

  private val maxUpdatesBeforeCommit = config.getLong("view.maxUpdatesBeforeCommit")
  private val analyzer = new StandardAnalyzer()
  private val iwc =  new IndexWriterConfig(analyzer)
  iwc.setOpenMode(OpenMode.CREATE_OR_APPEND)
  iwc.setRAMBufferSizeMB(config.getDouble("view.ramBufferSize"))
  managed(new IndexWriter(directory, new IndexWriterConfig(analyzer))) acquireAndGet {_ => ()}
  private val iwriter = new IndexWriter(directory, iwc)
  private val dirtyCount = new AtomicLong(0L)

  private var shutdownFlag = false
  private val autoCommitDelay = config.getDuration("view.autoCommitDelay").toMillis
  private val commitThread = new Thread() {
    override def run() = {
      while (!shutdownFlag) {
        Thread.sleep(autoCommitDelay)
        if (dirtyCount.get % maxUpdatesBeforeCommit != 0 && ! shutdownFlag) {
          dirtyCount.set(0L)
          iwriter.commit()
        }
      }
    }
  }.start()

  override def put(a: A): Unit = {
    iwriter.addDocument(mapper.toDoc(a))
    val dc = dirtyCount.incrementAndGet()
    if (dc % maxUpdatesBeforeCommit == 0) iwriter.commit()
  }

  override def replace(a: A): Unit = {
    delete(mapper.getId(a))
    put(a)
  }

  override def get(id: String): Option[String] = {
    val  reader = DirectoryReader.open(directory)
    val searcher = new IndexSearcher(reader)
    val hits = searcher.search(new TermQuery(new Term(mapper.IdField, id)), 1).scoreDocs
    hits.headOption.map(d => searcher.doc(d.doc).getField(mapper.SummaryField).stringValue())
  }

  override def getAsObj(id: String): Option[A] = {
    val reader = DirectoryReader.open(directory)
    val searcher = new IndexSearcher(reader)
    val hits = searcher.search(new TermQuery(new Term(mapper.IdField, id)), 1).scoreDocs
    hits.headOption.flatMap(d => mapper.fromDoc(searcher.doc(d.doc)).toOption)
  }

  override def query(field: String, value: String, n: Int): Seq[String] =  {
    val queryParser = new QueryParser(field, analyzer)
    val reader = DirectoryReader.open(directory)
    val searcher = new IndexSearcher(reader)
    val hits = searcher.search(queryParser.parse(QueryParserBase.escape(value)), n).scoreDocs
    hits.map(d => searcher.doc(d.doc).getField(mapper.SummaryField).stringValue())
  }

  override def queryAsObj(field: String, value: String, n: Int): Seq[A] = {
    val queryParser = new QueryParser(field, analyzer)
    val reader = DirectoryReader.open(directory)
    val searcher = new IndexSearcher(reader)
    val hits = searcher.search(queryParser.parse(QueryParserBase.escape(value)), n).scoreDocs
    hits.map(d => mapper.fromDoc(searcher.doc(d.doc))) collect {
      case Success(v) => v
    }
  }

  override def delete(id: String): Unit = {
    iwriter.deleteDocuments(new Term(mapper.IdField, id))
    val dc = dirtyCount.incrementAndGet()
    if (dc % maxUpdatesBeforeCommit == 0) iwriter.commit()
  }

  override def close(): Unit  = {
    shutdownFlag= true
    iwriter.close()
  }
}