package net.asynchorswim.ddd

import net.asynchorswim.ddd.ControlMessages.Shutdown
import akka.actor.{Actor, ActorContext, ActorLogging, Props}
import akka.persistence.AtLeastOnceDelivery

import scala.reflect.ClassTag
import scala.util.{Failure, Success, Try}

class TransientEntity[A <: Entity[A] : ClassTag] extends Actor with ActorLogging {

  override def preStart() = context.become(receive(create()))

  def receive = {
    case msg =>
      context.become(receive(create()))
      self forward msg
  }

  def receive(state: A): Receive = {
    case Shutdown =>context.stop(self)
    case StreamMessage(msg) =>
      processMessage(state, msg)
      sender ! StreamAck
    case msg =>
      processMessage(state, msg)
  }

  private def processMessage(state: A, msg: Any) = {
    val CommandResult(events, successSideEffect, failureSideEffect) = (state.receive orElse state.unhandled)(msg)
    Try((state /: events.asInstanceOf[Seq[Event]])((s, e) => s.applyEvent(e))) match {
      case Success(ns) =>
        msg match {
          case c: Commitable => c.commit()
          case _ =>
        }
        context.become(receive(ns))
        Try(successSideEffect()) match {
          case Success(_) =>
          case Failure(ex) =>
            log.warning(s"Could not apply side-effect(s): $successSideEffect")
            log.error(s"Failure: $ex")
        }
      case Failure(ex) =>
        failureSideEffect()
    }
  }

  private def create() : A = {
    val c = implicitly[scala.reflect.ClassTag[A]].runtimeClass
    Try(c.getConstructor(classOf[ActorContext])) match {
      case Success(ctor) =>
        ctor.newInstance(context).asInstanceOf[A]
      case Failure(_) =>
        c.newInstance().asInstanceOf[A]
    }
  }
}

object TransientEntity extends EntityPropsFactory {
  def props[A <: Entity[A] : ClassTag]: Props =
    Props(classOf[TransientEntity[A]], implicitly[ClassTag[A]])
}
