package net.asynchorswim.ddd

import akka.actor.{ActorContext, ActorLogging, Props, Stash}
import akka.persistence.AtLeastOnceDelivery.AtLeastOnceDeliverySnapshot
import akka.persistence.{AtLeastOnceDelivery, PersistentActor, RecoveryCompleted, SnapshotOffer}
import net.asynchorswim.ddd.ControlMessages.{Shutdown, TakeSnapshot}
import scala.reflect.ClassTag
import scala.util.{Failure, Success, Try}

class EventSourcedEntity[A <: Entity[A] : ClassTag] extends PersistentActor with ActorLogging with Stash with AtLeastOnceDelivery {

  override def persistenceId = self.path.parent.name + "-" + self.path.name

  def receiveRecover: Receive = {
    case e =>
      context.become(receiveRecover(create()))
      self forward e
  }

  override def receiveCommand = {
    case _ => log.error("Wiring error: Received message in stateless receive() method")
  }

  def receiveRecover(state: A): Receive = {
    case ALORequest(target, message) => deliver(target) { deliveryId => ALOEnvelope(deliveryId, message) }
    case Ack(id) => confirmDelivery(id)
    case event: Event =>
      Try(state.applyEvent(event)) match {
        case Success(ns) =>
          context.become(receiveRecover(ns))
        case Failure(ex) =>
          log.error(s"Error applying event: $event")
          log.error(s"Failure: $ex")
      }
    case SnapshotOffer(_, snapshot) =>
      val ss = snapshot.asInstanceOf[(AtLeastOnceDeliverySnapshot, A)]
      setDeliverySnapshot(ss._1)
      context.become(receiveRecover(ss._2))
    case RecoveryCompleted =>
      state.postRecover()
      unstashAll()
      context.become(receiveCommand(state))
    case e =>
      stash()
  }

  def receiveCommand(state: A): Receive = {
    case TakeSnapshot => saveSnapshot((getDeliverySnapshot, state))
    case Shutdown => context.stop(self)
    case a @ Ack(id) => persist(a) { ack => confirmDelivery(id) }
    case a @ ALORequest(target, message) => persist(a) { req =>
      deliver(target) { deliveryId => ALOEnvelope(deliveryId, message) }
    }
    case ALOEnvelope(deliveryId, msg) =>
      processCommand(state, msg)
      confirmDelivery(deliveryId)
    case StreamMessage(msg) =>
      processCommand(state, msg)
      sender ! StreamAck
    case msg =>
      processCommand(state, msg)
  }

  private def processCommand(state: A, msg: Any) = {
    val CommandResult(events, successSideEffect, failureSideEffect) = (state.receive orElse state.unhandled)(msg)
    if (events.nonEmpty) {
      Try {
        persistAll[Event](events) { e => }
        context.become(receiveCommand((state /: events)(_.applyEvent(_))))
        applySideEffect(successSideEffect)
      } recover {
        case _ => applySideEffect(failureSideEffect)
      }
    }
    else {
      msg match {
        case c: Commitable => c.commit()
        case _ =>
      }
      applySideEffect(successSideEffect)
    }
  }

  private def applySideEffect(sideEffect: () => Unit) =
    Try(sideEffect()) match {
      case Success(_) =>
      case Failure(ex) =>
        log.warning(s"Could not apply side-effect: $sideEffect")
        log.error(s"Failure: $ex")
    }

  private def create() : A = {
    val c = implicitly[ClassTag[A]].runtimeClass
    Try(c.getConstructor(classOf[ActorContext])) match {
      case Success(ctor) =>
        ctor.newInstance(context).asInstanceOf[A]
      case Failure(_) =>
        c.newInstance().asInstanceOf[A]
    }
  }
}

object EventSourcedEntity extends EntityPropsFactory {
  def props[A <: Entity[A] : ClassTag]: Props =
    Props(classOf[EventSourcedEntity[A]], implicitly[ClassTag[A]])
}
