package net.asynchorswim.ddd

import language.postfixOps
import akka.actor.{Actor, ActorLogging, ActorRef}
import scala.reflect.ClassTag
import akka.pattern.ask
import akka.util.Timeout
import concurrent.duration._
import scala.concurrent.duration.Duration
import scala.concurrent.{Await, Future}

abstract class AbstractAggregateRoot[A <: Entity[A] : ClassTag](propsFactory: EntityPropsFactory) extends Actor with ActorLogging {

  val route: PartialFunction[Any, String]
  val payload: PartialFunction[Any, Any] = { case x => x }

  implicit private val ex = context.dispatcher
  implicit val timeout = Timeout(10 seconds)

  override def preStart() = context.become(receive(Map.empty[String, ActorRef]))

  def receive = {
    case msg =>
      context.become(receive(Map.empty[String, ActorRef]))
      self forward payload(msg)
  }

  def receive(state: Map[String, ActorRef]): Receive = {
    case msg: Broadcast =>
      state.values.foreach(_ forward msg)
    case msg  =>
      processMessage(state, msg)
  }

  private def processMessage(state: Map[String, ActorRef], msg: Any) = {
    val id = msg match {
      case StreamMessage(m) => route(m)
      case ALOEnvelope(_, m) => route(m)
      case m => route(m)
    }
    val pload = msg match {
      case StreamMessage(m) => StreamMessage(payload(m))
      case ALOEnvelope(i, m) => ALOEnvelope(i, payload(m))
      case m => payload(m)
    }
    state.get(id) match {
      case Some(actor) =>
        actor forward pload
      case None if pload.isInstanceOf[CanBeFirst] =>
        val actor = context.actorOf(propsFactory.props[A], id)
        context.become(receive(state.updated(id, actor)))
        actor forward pload
      case _ =>
        sender ! AggregateRoot.UnknownEntity(id)
    }
  }
}

object AggregateRoot {
  case class UnknownEntity(id: String)
}

abstract class TransientAggregateRoot[A <: Entity[A] : ClassTag] extends AbstractAggregateRoot[A](TransientEntity)

abstract class EventSourcedAggregateRoot[A <: Entity[A] : ClassTag] extends AbstractAggregateRoot[A](EventSourcedEntity)

