package net.asynchorswim.ddd

import akka.actor.{ActorRef, ActorLogging, Actor}
import scala.reflect.ClassTag

abstract class AbstractAggregateRoot[A <: Entity[A] : ClassTag](propsFactory: EntityPropsFactory) extends Actor with ActorLogging {

  val route: PartialFunction[Any, String]
  val payload: PartialFunction[Any, Any] = { case x => x }

  override def preStart() = context.become(receive(Map.empty[String, ActorRef]))

  def receive = {
    case msg =>
      context.become(receive(Map.empty[String, ActorRef]))
      self forward payload(msg)
  }

  def receive(state: Map[String, ActorRef]): Receive = {
    case msg: Broadcast =>
      state.values.foreach(_ forward payload(msg))
    case msg  =>
      val id = route(msg)
      state.get(id) match {
        case Some(actor) =>
          actor forward payload(msg)
        case None if payload(msg).isInstanceOf[CanBeFirst] =>
          val actor = context.actorOf(propsFactory.props[A], id)
          context.become(receive(state.updated(id, actor)))
          actor forward payload(msg)
        case _ =>
          sender ! AggregateRoot.UnknownEntity(id)
      }
  }
}

object AggregateRoot {
  case class UnknownEntity(id: String)
}

abstract class TransientAggregateRoot[A <: Entity[A] : ClassTag] extends AbstractAggregateRoot[A](TransientEntity)

abstract class EventSourcedAggregateRoot[A <: Entity[A] : ClassTag] extends AbstractAggregateRoot[A](EventSourcedEntity)

