/*
 * Copyright (C) 2016, apexes.net. All rights reserved.
 * 
 *        http://www.apexes.net
 * 
 */
package net.apexes.wsonrpc.server;

import net.apexes.wsonrpc.core.ServiceRegistry;
import net.apexes.wsonrpc.core.WebSocketSession;
import net.apexes.wsonrpc.core.WsonrpcConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;

/**
 * 
 * @author <a href="mailto:hedyn@foxmail.com">HeDYn</a>
 *
 */
public class WsonrpcServerBase implements WsonrpcServer {

    private static final Logger LOG = LoggerFactory.getLogger(WsonrpcServerBase.class);

    private final WsonrpcServerEngine wsonrpcEngine;
    private WsonrpcServerListener serverListener;

    public WsonrpcServerBase(WsonrpcConfig config) {
        wsonrpcEngine = new WsonrpcServerEngine(config);
    }

    @Override
    public ServiceRegistry getServiceRegistry() {
        return wsonrpcEngine.getServiceRegistry();
    }

    @Override
    public void setServerListener(WsonrpcServerListener listener) {
        this.serverListener = listener;
    }

    @Override
    public void setWsonrpcRequestInterceptor(WsonrpcRequestInterceptor interceptor) {
        wsonrpcEngine.setWsonrpcRequestInterceptor(interceptor);
    }

    /**
     * Client端已经连接上
     * @param session
     */
    public void onOpen(WebSocketSession session) {
        WsonrpcRemotes.addRemote(session, new RemoteWsonrpcEndpoint(session, wsonrpcEngine));
        fireOpen(session);
    }

    /**
     * Client端被关闭了
     * @param sessionId
     */
    public void onClose(String sessionId) {
        WsonrpcRemotes.removeRemote(sessionId);
        fireClose(sessionId);
    }

    /**
     * 收到Client端发来的数据
     * @param sessionId
     * @param buffer
     */
    public void onMessage(String sessionId, ByteBuffer buffer) {
        WebSocketSession session = WsonrpcRemotes.getSession(sessionId);
        byte[] bytes = buffer.array();
        try {
            fireMessage(sessionId, bytes);
            wsonrpcEngine.handle(session, bytes);
        } catch (Exception ex) {
            onError(session.getId(), ex);
        }
    }
    
    public void onError(String sessionId, Throwable error) {
        wsonrpcEngine.onError(sessionId, error);
    }
    
    private void fireOpen(WebSocketSession session) {
        LOG.debug("onOpen : {}", session.getId());
        if (serverListener != null) {
            serverListener.onOpen(session);
        }
    }

    private void fireClose(String sessionId) {
        LOG.debug("onClose : {}", sessionId);
        if (serverListener != null) {
            serverListener.onClose(sessionId);
        }
    }
    
    private void fireMessage(String sessionId, byte[] bytes) {
        LOG.debug("onMessage : {}", sessionId);
        if (serverListener != null) {
            serverListener.onMessage(sessionId, bytes);
        }
    }

}
