/* 
 * Copyright (c) 2013 Amr Hassan <amr.hassan@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining 
 * a copy of this software and associated documentation files (the 
 * "Software"), to deal in the Software without restriction, including 
 * without limitation the rights to use, copy, modify, merge, publish, 
 * distribute, sublicense, and/or sell copies of the Software, and to 
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be 
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


package net.amrhassan.sqlbuilder.sources;

import net.amrhassan.sqlbuilder.restrictions.SqlRestriction;

/**
 * A one-stop place for all the possible SQL sources
 */
public final class SqlSources {
	
	private SqlSources() {
		// Private constructor to prevent instantiation
	}
	
	/**
	 * A simple JOIN source where the restriction is used for the ON clause
	 */
	public static SqlSource join(TableSqlSource table1, TableSqlSource table2, SqlRestriction on) {
		return new JoinSqlSource(table1, table2, on);
	}
	
	/**
	 * A LEFT JOIN source where the restriction is used for the ON clause 
	 */
	public static SqlSource leftJoin(TableSqlSource table1, TableSqlSource table2, SqlRestriction on) {
		return new JoinSqlSource(JoinSqlSource.Type.LEFT, table1, table2, on);
	}
	
	/**
	 * A RIGHT JOIN where the restriction is used for the ON clause
	 */
	public static SqlSource rightJoin(TableSqlSource table1, TableSqlSource table2, SqlRestriction on) {
		return new JoinSqlSource(JoinSqlSource.Type.RIGHT, table1, table2, on);
	}
	
	/**
	 * A FULL JOIN where the restriction is used for the ON clause
	 */
	public static SqlSource fullJoin(TableSqlSource table1, TableSqlSource table2, SqlRestriction on) {
		return new JoinSqlSource(JoinSqlSource.Type.FULL, table1, table2, on);
	}
	
	/**
	 * A table source
	 * 
	 * @param name The name of the table
	 */
	public static SqlSource table(String name) {
		return new TableSqlSource(name);
	}
	
	/**
	 * A table source
	 * 
	 * @param name The name of the table
	 * @param alias The alias for the table
	 */
	public static TableSqlSource table(String name, String alias) {
		return new TableSqlSource(name, alias);
	}
}
