/* 
 * Copyright (c) 2013 Amr Hassan <amr.hassan@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining 
 * a copy of this software and associated documentation files (the 
 * "Software"), to deal in the Software without restriction, including 
 * without limitation the rights to use, copy, modify, merge, publish, 
 * distribute, sublicense, and/or sell copies of the Software, and to 
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be 
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.amrhassan.sqlbuilder.restrictions;

import java.util.Arrays;
import java.util.Collection;

import net.amrhassan.sqlbuilder.SelectSqlStatement;

/**
 * A one-stop place for all the possible SQL restrictions.
 * 
 * Most restrictions are provided in two flavours, one that
 * automatically sets the restriction operands to the '?'
 * character, the other needs the operands to be specified
 * explicitly as Strings.
 */
public final class SqlRestrictions {
	
	private SqlRestrictions() {
		// Private constructor to prevent instantiation
	}
	
	/**
	 * Equality between the provided field name and '?'
	 */
	public static SqlRestriction eq(String field) {
		return new EqualsSqlRestriction(field, "?");
	}
	
	/**
	 * Equality between two operands (lhs = rhs).
	 * 
	 * Be aware that the provided operands will not be escaped. Use the
	 * no-argument version of this method which places an automatic
	 * '?' character where each operand should go so that it's suitable
	 * for using with a {@link PreparedStatement}. 
	 */
	public static SqlRestriction eq(String lhs, String rhs) {
		return new EqualsSqlRestriction(lhs, rhs);
	}
	
	/**
	 * AND combination of restrictions
	 */
	public static SqlRestriction and(SqlRestriction... restrictions) {
		return new AndSqlRestriction(Arrays.asList(restrictions));
	}
	
	/**
	 * AND combination of restrictions
	 */
	public static SqlRestriction and(Collection<SqlRestriction> restrictions) {
		if (restrictions.isEmpty())
			return null;
		
		return new AndSqlRestriction(restrictions);
	}
	
	/**
	 * The specified field falls inside a range; (field BETWEEN ? AND ?).
	 */
	public static SqlRestriction between(String field) {
		return new BetweenSqlRestriction(field, "?", "?");
	}
	
	/**
	 * The specified field falls inside a range; (field BETWEEN lowerBound AND upperBound).
	 * 
	 * Be aware that the provided operands will not be escaped; Use the
	 * no-argument version of this method which places an automatic
	 * '?' character where each operand should go so that it's suitable
	 * for using with a {@link PreparedStatement} 
	 */
	public static SqlRestriction between(String field, String lowerBound, String upperBound) {
		return new BetweenSqlRestriction(field, lowerBound, upperBound);
	}
	
	/**
	 * The specified field is in a list of provided parameters; (field IN (?,?,?,...))
	 * 
	 * @param field The field name or its alias
	 * @param operandCount The number of operands in the list of possible values
	 */
	public static SqlRestriction in(String field, int operandCount) {
		String[] values = new String[operandCount];
		
		Arrays.fill(values, "?");
		
		return new InSqlRestriction(field, values);
	}
	
	/**
	 * The specified field lies in the result of the provided sub-query
	 * 
	 * @param field The field name or its alias
	 * @param selectStatement The sub-query
	 */
	public static SqlRestriction in(String field, SelectSqlStatement selectStatement) {
		return new InSqlRestriction(field, new String[] {selectStatement.toString()});
	}
	
	/**
	 * The specified field is less-than-or-equal to another operand specified
	 * by a '?' character
	 * 
	 * @param field The field name or its alias
	 */
	public static SqlRestriction lte(String field) {
		return new LessThanOrEqualSqlRestriction(field, "?");
	}
	
	/**
	 * The specified field is less-than another operand specified
	 * by a '?' character
	 * 
	 * @param field The field name or its alias
	 */
	public static SqlRestriction lt(String field) {
		return new LessThanSqlRestriction(field, "?");
	}
	
	/**
	 * The specified field is greater-than-or-equal to another operand specified
	 * by a '?' character
	 * 
	 * @param field The field name or its alias
	 */
	public static SqlRestriction gte(String field) {
		return new GreaterThanOrEqualSqlRestriction(field, "?");
	}
	
	/**
	 * The specified field is greater-than another operand specified
	 * by a '?' character
	 * 
	 * @param field
	 * @return
	 */
	public static SqlRestriction gt(String field) {
		return new GreaterThanSqlRestriction(field, "?");
	}
}
