/* 
 * Copyright (c) 2013 Amr Hassan <amr.hassan@gmail.com>
 * 
 * Permission is hereby granted, free of charge, to any person obtaining 
 * a copy of this software and associated documentation files (the 
 * "Software"), to deal in the Software without restriction, including 
 * without limitation the rights to use, copy, modify, merge, publish, 
 * distribute, sublicense, and/or sell copies of the Software, and to 
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be 
 * included in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF 
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE 
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package net.amrhassan.sqlbuilder;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;

import net.amrhassan.sqlbuilder.restrictions.SqlRestriction;
import net.amrhassan.sqlbuilder.sources.SqlSource;
import net.amrhassan.sqlbuilder.utils.StringUtils;

/**
 * A SELECT SQL statement
 */
public final class SelectSqlStatement implements SqlStatement {
	
	/**
	 * The order specified in an ORDER BY clause
	 */
	public enum Order {
		ASCENDING, DESCENDING;
	};
	
	private boolean distinct = false;
	
	private List<String> fields = new LinkedList<String>();
	
	private List<SqlSource> sources = new LinkedList<SqlSource>();
	
	private SqlRestriction restriction;
	
	private List<String> orderByFields = new LinkedList<String>();
	private Order order;
	
	private List<String> groupByFields = new LinkedList<String>();
	
	/**
	 * Enable the DISTINCT clause
	 */
	public void enableDistinct() {
		this.distinct = true;
	}
	
	/**
	 * Disable the DISTINCT clause
	 */
	public void disableDistinct() {
		this.distinct = false;
	}
	
	/**
	 * Add a source for the statement, which can be a simple table name
	 * or a complex JOIN clause (see {@link SqlSources})
	 */
	public void addSource(SqlSource source) {
		sources.add(source);
	}
	
	/**
	 * Add a field to be selected. Fields can be column names or aggregate function
	 * expressions.
	 * 
	 * @param name Column name or aggregate function expression
	 * @param alias An alias for the specified field name
	 */
	public void addFieldAs(String name, String alias) {
		fields.add(String.format("%s AS %s", name, alias));
	}

	/**
	 * Add a fields to be selected. Fields can be column names or aggregate function
	 * expressions.
	 * 
	 * @param names Column names or aggregate function expression
	 */
	public void addFields(String... names) {
		for (String name : names)
			fields.add(name);
	}
	
	/**
	 * Specify fields to be used for the ORDER BY clause
	 * 
	 * @param fields The field names or their aliases if they were specified with aliases
	 */
	public void addOrderBy(String... fields) {
		orderByFields.addAll(Arrays.asList(fields));
	}
	
	/**
	 * Specify a field to be used for the ORDER BY clause
	 * 
	 * @param fields The field names or their aliases if they were specified with aliases 
	 * @param order The order of the rows
	 */
	public void addOrderBy(Order order, String... fields) {
		addOrderBy(fields);
		this.order = order;
	}
	
	/**
	 * Add a field to be used for the GROUP BY clause
	 * @param field The field name or its alias if it was specified with an alias
	 */
	public void addGroupByFields(String... fields) {
		for (String field : fields)
			groupByFields.add(field);
	}
	
	/**
	 * The restrictions used in the WHERE clause 
	 * 
	 * @param restriction (See {@link SqlSources})
	 */
	public void setRestriction(SqlRestriction restriction) {
		this.restriction = restriction;
	}
	
	@Override
	public String toString() {
		
		List<String> segments = new LinkedList<String>();
		
		segments.add("SELECT");
		
		if (distinct)
			segments.add("DISTINCT");
		
		segments.add(StringUtils.join(", ", fields));
		segments.add("FROM");
		segments.add(StringUtils.join(", ", sources));
		
		if (restriction != null) {
			segments.add("WHERE");
			segments.add(restriction.toString());
		}
		
		if (!orderByFields.isEmpty()) {
			segments.add("ORDER BY");
			segments.add(StringUtils.join(", ", orderByFields));
			
			if (order != null) {
				if (order == Order.DESCENDING)
					segments.add("DESC");
				else
					segments.add("ASC");
			}
		}
		
		if (!groupByFields.isEmpty()) {
			segments.add("GROUP BY");
			segments.add(StringUtils.join(", ", groupByFields));
		}
		
		return StringUtils.join(" ", segments) + ";";
	}
}
