/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017-2024 Daniel Alievsky, AlgART Laboratory (http://algart.net)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package net.algart.executors.modules.opencv.matrices.video;

import net.algart.executors.modules.opencv.common.VoidResultUMatFilter;
import net.algart.executors.modules.opencv.util.O2SMat;
import net.algart.executors.api.Port;
import org.bytedeco.opencv.opencv_core.*;
import org.bytedeco.opencv.opencv_video.BackgroundSubtractor;

public abstract class AbstractBackgroundSubtractor extends VoidResultUMatFilter {
    public static final String OUTPUT_BACKGROUND = "background";

    private boolean reset = true;
    private boolean detectShadows = true;

    private volatile BackgroundSubtractor subtractor = null;
    private volatile Mat background = null;

    public boolean isReset() {
        return reset;
    }

    public void setReset(boolean reset) {
        this.reset = reset;
    }

    public AbstractBackgroundSubtractor() {
        addOutputMat(OUTPUT_BACKGROUND);
    }

    public boolean isDetectShadows() {
        return detectShadows;
    }

    public void setDetectShadows(boolean detectShadows) {
        this.detectShadows = detectShadows;
    }

    public Mat getBackground() {
        return background;
    }

    /*Repeat() \bMat ==> UMat */
    @Override
    public void process(Mat result, Mat source) {
        final Port backgroundPort = getOutputPort(OUTPUT_BACKGROUND);
        if (subtractor == null || reset) {
            if (subtractor != null) {
                subtractor.close();
            }
            subtractor = createBackgroundSubtractor();
        }
        subtractor.apply(source, result);
        if (backgroundPort != null && backgroundPort.isConnected()) {
            if (background != null) {
                background.close();
            }
            createBackground();
            subtractor.getBackgroundImage(background);
            O2SMat.setTo(getMat(OUTPUT_BACKGROUND), background);
        }
    }

    /*Repeat.AutoGeneratedStart !! Auto-generated: NOT EDIT !! */
    @Override
    public void process(UMat result, UMat source) {
        final Port backgroundPort = getOutputPort(OUTPUT_BACKGROUND);
        if (subtractor == null || reset) {
            if (subtractor != null) {
                subtractor.close();
            }
            subtractor = createBackgroundSubtractor();
        }
        subtractor.apply(source, result);
        if (backgroundPort != null && backgroundPort.isConnected()) {
            if (background != null) {
                background.close();
            }
            createBackground();
            subtractor.getBackgroundImage(background);
            O2SMat.setTo(getMat(OUTPUT_BACKGROUND), background);
        }
    }

    /*Repeat.AutoGeneratedEnd*/

    @Override
    public void close() {
        if (subtractor != null) {
            subtractor.close();
            subtractor = null;
        }
        if (background != null) {
            background.close();
            background = null;
        }
        super.close();
    }

    abstract BackgroundSubtractor createBackgroundSubtractor();

    private void createBackground() {
        background = new Mat();
    }
}
