/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017-2024 Daniel Alievsky, AlgART Laboratory (http://algart.net)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package net.algart.executors.modules.cv.matrices.objects.binary.boundaries;

import net.algart.matrices.scanning.Boundary2DProjectionMeasurer;

public enum ThirdProjectionValue {
    /**
     * No 3rd number (only 2).
     */
    NONE {
        @Override
        Double thirdProjectionValue(Boundary2DProjectionMeasurer measurer, int projectionIndex, double pixelSize) {
            return null;
        }
    },

    /**
     * The 3rd number in any projection port will be the angle.
     */
    ANGLE {
        @Override
        Double thirdProjectionValue(Boundary2DProjectionMeasurer measurer, int projectionIndex, double pixelSize) {
            return normalizeGreater180(Math.toDegrees(measurer.startAngleInRadians())
                    + projectionIndex * 180.0 / measurer.numberOfDirections());
        }
    },

    /**
     * The 3rd number in any projection port will be the index of this projection.
     */
    INDEX {
        @Override
        Double thirdProjectionValue(Boundary2DProjectionMeasurer measurer, int projectionIndex, double pixelSize) {
            return (double) projectionIndex;
        }
    };

    // Really pixelSize is not used by current implementations
    abstract Double thirdProjectionValue(Boundary2DProjectionMeasurer measurer, int projectionIndex, double pixelSize);

    public int oneIfExist() {
        return this == NONE ? 0 : 1;
    }

    private static double normalizeGreater180(double greater180) {
        return greater180 >= 180.0 ? greater180 - 180.0 : greater180;
    }
}
