/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017-2023 Daniel Alievsky, AlgART Laboratory (http://algart.net)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package net.algart.executors.modules.cv.matrices.objects.labels;

import java.util.Arrays;
import java.util.Objects;

abstract class FirstNonZeroCalculator extends CardinalitiesCalculator {
    final int numberOfChannels;
    final int[][] threadFirstNonZeroIndexesIncreased;
    private final int[][] requestedFirstNonZeroIndexes;
    int[] firstNonZeroIndexes;
    int[] firstNonZeroIntValues;
    float[] firstNonZeroFloatValues;
    // Only one last 2 arrays is non-null.
    // Note: first non-zero values are ordered as RGBRGB...

    FirstNonZeroCalculator(int[] labels, int numberOfChannels) {
        super(labels);
        this.numberOfChannels = numberOfChannels;
        this.requestedFirstNonZeroIndexes = requestClearedIntArrays(numberOfTasks());
        this.threadFirstNonZeroIndexesIncreased = requestedFirstNonZeroIndexes.clone();
    }

    static FirstNonZeroCalculator getInstance(int[] labels, Object[] channels) {
        Objects.requireNonNull(labels, "Null labels");
        Objects.requireNonNull(channels, "Null channels");
        if (channels.length == 0) {
            throw new IllegalArgumentException("Empty channels array");
        }
        final Object channel0 = channels[0];
        if (!isArraySupported(channel0)) {
            throw new IllegalArgumentException("Illegal array type: " + channel0);
        }
        for (int k = 1; k < channels.length; k++) {
            if (channels[k].getClass() != channel0.getClass()) {
                throw new IllegalArgumentException("Different type of channels: " + channels[k].getClass()
                        + " != " + channel0.getClass());
            }
        }
        switch (channels.length) {
            /*Repeat() case 1  ==> case 2,,case 3;;
                       1(Channels) ==> 2$1,,3$1
             */
            case 1: {
                if (channel0 instanceof byte[]) {
                    return new FirstNonZeroCalculator1Channels.ForBytes(labels, castToByte(channels));
                } else if (channel0 instanceof short[]) {
                    return new FirstNonZeroCalculator1Channels.ForShorts(labels, castToShort(channels));
                } else if (channel0 instanceof int[]) {
                    return new FirstNonZeroCalculator1Channels.ForInts(labels, castToInt(channels));
                } else if (channel0 instanceof float[]) {
                    return new FirstNonZeroCalculator1Channels.ForFloats(labels, castToFloat(channels));
                } else if (channel0 instanceof double[]) {
                    return new FirstNonZeroCalculator1Channels.ForDoubles(labels, castToDouble(channels));
                } else {
                    throw new AssertionError();
                }
            }
            /*Repeat.AutoGeneratedStart !! Auto-generated: NOT EDIT !! */
            case 2: {
                if (channel0 instanceof byte[]) {
                    return new FirstNonZeroCalculator2Channels.ForBytes(labels, castToByte(channels));
                } else if (channel0 instanceof short[]) {
                    return new FirstNonZeroCalculator2Channels.ForShorts(labels, castToShort(channels));
                } else if (channel0 instanceof int[]) {
                    return new FirstNonZeroCalculator2Channels.ForInts(labels, castToInt(channels));
                } else if (channel0 instanceof float[]) {
                    return new FirstNonZeroCalculator2Channels.ForFloats(labels, castToFloat(channels));
                } else if (channel0 instanceof double[]) {
                    return new FirstNonZeroCalculator2Channels.ForDoubles(labels, castToDouble(channels));
                } else {
                    throw new AssertionError();
                }
            }

            case 3: {
                if (channel0 instanceof byte[]) {
                    return new FirstNonZeroCalculator3Channels.ForBytes(labels, castToByte(channels));
                } else if (channel0 instanceof short[]) {
                    return new FirstNonZeroCalculator3Channels.ForShorts(labels, castToShort(channels));
                } else if (channel0 instanceof int[]) {
                    return new FirstNonZeroCalculator3Channels.ForInts(labels, castToInt(channels));
                } else if (channel0 instanceof float[]) {
                    return new FirstNonZeroCalculator3Channels.ForFloats(labels, castToFloat(channels));
                } else if (channel0 instanceof double[]) {
                    return new FirstNonZeroCalculator3Channels.ForDoubles(labels, castToDouble(channels));
                } else {
                    throw new AssertionError();
                }
            }
            /*Repeat.AutoGeneratedEnd*/
            default: {
                if (channel0 instanceof byte[]) {
                    return new ForBytes(labels, castToByte(channels));
                } else if (channel0 instanceof short[]) {
                    return new ForShorts(labels, castToShort(channels));
                } else if (channel0 instanceof int[]) {
                    return new ForInts(labels, castToInt(channels));
                } else if (channel0 instanceof float[]) {
                    return new ForFloats(labels, castToFloat(channels));
                } else if (channel0 instanceof double[]) {
                    return new ForDoubles(labels, castToDouble(channels));
                } else {
                    throw new AssertionError();
                }
            }
        }
    }

    @Override
    public void close() {
        super.close();
        releaseAndClearIntArrays(requestedFirstNonZeroIndexes, maxLabel + 1);
    }

    @Override
    protected void finish() {
        super.finish();
        if ((long) maxLabel * (long) numberOfChannels > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Too large required array for " + numberOfChannels
                    + " channels: more that 2^31-1 elements");
        }
        this.firstNonZeroIndexes = new int[maxLabel];
        Arrays.fill(this.firstNonZeroIndexes, -1);
        for (int[] increasedIndexes : this.threadFirstNonZeroIndexesIncreased) {
            final int length = Math.min(this.firstNonZeroIndexes.length, increasedIndexes.length - 1);
            for (int label = 0; label < length; label++) {
                final int increasedIndex = increasedIndexes[label + 1];
                // label+1: in the resulting "this.firstNonZeroIndexes" we use zero element
                // to store information about the object with label #1 (zero labels are not analyzed at all).
                // So, actual information for the label must be place in this.firstNonZeroIndexes[label-1]
                final int index = increasedIndex - 1;
                // increasedIndex is an index in the matrix (0..labels.length-1), INCREASED by 1 inside
                // processSubArr methods - to allow 0 to be a "reserved" value and to avoid extra efforts
                // for initializing newly allocated arrays by non-zero value.
                final int previous = this.firstNonZeroIndexes[label];
                if (index >= 0 && (previous == -1 || index < previous)) {
                    this.firstNonZeroIndexes[label] = index;
                }
            }
        }
    }

    /*Repeat() Bytes  ==> Shorts,,Ints,,Floats,,Doubles;;
               byte   ==> short,,int,,float,,double;;
               (data\[c\]\[\w*\]) \& 0xFF ==> $1 & 0xFFFF,,$1,,$1,,(float) $1;;
               (firstNonZeroIntValues) ==> $1,,$1,,firstNonZeroFloatValues,,...;;
               (new int\[) ==> $1,,$1,,new float[,,...
     */
    private static class ForBytes extends FirstNonZeroCalculator {
        private final byte[][] data;

        public ForBytes(int[] labels, byte[][] data) {
            super(labels, data.length);
            this.data = data;
        }

        @Override
        protected void processSubArr(int p, int count, int threadIndex) {
            int[] cardinalities = this.threadCardinalities[threadIndex];
            int[] firstNonZeroIndexesIncreased = this.threadFirstNonZeroIndexesIncreased[threadIndex];
            for (int k = p, kMax = k + count; k < kMax; k++) {
                int label = labels[k];
                if (label > 0) {
                    if (label >= cardinalities.length) {
                        cardinalities = ensureCapacityForLabel(cardinalities, label);
                        firstNonZeroIndexesIncreased = ensureCapacityForLabel(firstNonZeroIndexesIncreased, label);
                    }
                    cardinalities[label]++;
                    if (firstNonZeroIndexesIncreased[label] == 0) {
                        boolean nonZero = false;
                        for (int c = 0; c < numberOfChannels; c++) {
                            if (data[c][k] != 0) {
                                nonZero = true;
                                break;
                            }
                        }
                        if (nonZero) {
                            firstNonZeroIndexesIncreased[label] = k + 1;
                            // - firstNonZeroIndexesIncreased must contain NON-ZERO values to avoid extra time and
                            // efforts for initializing newly allocated arrays by -1 or something like this
                        }
                    }
                }
            }
            this.threadCardinalities[threadIndex] = cardinalities;
            this.threadFirstNonZeroIndexesIncreased[threadIndex] = firstNonZeroIndexesIncreased;
        }

        @Override
        protected void finish() {
            super.finish();
            this.firstNonZeroIntValues = new int[firstNonZeroIndexes.length * numberOfChannels];
            // - zero-filled by Java
            for (int label = 0, disp = 0; label < firstNonZeroIndexes.length; label++, disp += numberOfChannels) {
                final int index = firstNonZeroIndexes[label];
                if (index >= 0) {
                    for (int c = 0; c < numberOfChannels; c++) {
                        this.firstNonZeroIntValues[disp + c] = data[c][index] & 0xFF;
                    }
                }
            }
        }
    }

    /*Repeat.AutoGeneratedStart !! Auto-generated: NOT EDIT !! */
    private static class ForShorts extends FirstNonZeroCalculator {
        private final short[][] data;

        public ForShorts(int[] labels, short[][] data) {
            super(labels, data.length);
            this.data = data;
        }

        @Override
        protected void processSubArr(int p, int count, int threadIndex) {
            int[] cardinalities = this.threadCardinalities[threadIndex];
            int[] firstNonZeroIndexesIncreased = this.threadFirstNonZeroIndexesIncreased[threadIndex];
            for (int k = p, kMax = k + count; k < kMax; k++) {
                int label = labels[k];
                if (label > 0) {
                    if (label >= cardinalities.length) {
                        cardinalities = ensureCapacityForLabel(cardinalities, label);
                        firstNonZeroIndexesIncreased = ensureCapacityForLabel(firstNonZeroIndexesIncreased, label);
                    }
                    cardinalities[label]++;
                    if (firstNonZeroIndexesIncreased[label] == 0) {
                        boolean nonZero = false;
                        for (int c = 0; c < numberOfChannels; c++) {
                            if (data[c][k] != 0) {
                                nonZero = true;
                                break;
                            }
                        }
                        if (nonZero) {
                            firstNonZeroIndexesIncreased[label] = k + 1;
                            // - firstNonZeroIndexesIncreased must contain NON-ZERO values to avoid extra time and
                            // efforts for initializing newly allocated arrays by -1 or something like this
                        }
                    }
                }
            }
            this.threadCardinalities[threadIndex] = cardinalities;
            this.threadFirstNonZeroIndexesIncreased[threadIndex] = firstNonZeroIndexesIncreased;
        }

        @Override
        protected void finish() {
            super.finish();
            this.firstNonZeroIntValues = new int[firstNonZeroIndexes.length * numberOfChannels];
            // - zero-filled by Java
            for (int label = 0, disp = 0; label < firstNonZeroIndexes.length; label++, disp += numberOfChannels) {
                final int index = firstNonZeroIndexes[label];
                if (index >= 0) {
                    for (int c = 0; c < numberOfChannels; c++) {
                        this.firstNonZeroIntValues[disp + c] = data[c][index] & 0xFFFF;
                    }
                }
            }
        }
    }


    private static class ForInts extends FirstNonZeroCalculator {
        private final int[][] data;

        public ForInts(int[] labels, int[][] data) {
            super(labels, data.length);
            this.data = data;
        }

        @Override
        protected void processSubArr(int p, int count, int threadIndex) {
            int[] cardinalities = this.threadCardinalities[threadIndex];
            int[] firstNonZeroIndexesIncreased = this.threadFirstNonZeroIndexesIncreased[threadIndex];
            for (int k = p, kMax = k + count; k < kMax; k++) {
                int label = labels[k];
                if (label > 0) {
                    if (label >= cardinalities.length) {
                        cardinalities = ensureCapacityForLabel(cardinalities, label);
                        firstNonZeroIndexesIncreased = ensureCapacityForLabel(firstNonZeroIndexesIncreased, label);
                    }
                    cardinalities[label]++;
                    if (firstNonZeroIndexesIncreased[label] == 0) {
                        boolean nonZero = false;
                        for (int c = 0; c < numberOfChannels; c++) {
                            if (data[c][k] != 0) {
                                nonZero = true;
                                break;
                            }
                        }
                        if (nonZero) {
                            firstNonZeroIndexesIncreased[label] = k + 1;
                            // - firstNonZeroIndexesIncreased must contain NON-ZERO values to avoid extra time and
                            // efforts for initializing newly allocated arrays by -1 or something like this
                        }
                    }
                }
            }
            this.threadCardinalities[threadIndex] = cardinalities;
            this.threadFirstNonZeroIndexesIncreased[threadIndex] = firstNonZeroIndexesIncreased;
        }

        @Override
        protected void finish() {
            super.finish();
            this.firstNonZeroIntValues = new int[firstNonZeroIndexes.length * numberOfChannels];
            // - zero-filled by Java
            for (int label = 0, disp = 0; label < firstNonZeroIndexes.length; label++, disp += numberOfChannels) {
                final int index = firstNonZeroIndexes[label];
                if (index >= 0) {
                    for (int c = 0; c < numberOfChannels; c++) {
                        this.firstNonZeroIntValues[disp + c] = data[c][index];
                    }
                }
            }
        }
    }


    private static class ForFloats extends FirstNonZeroCalculator {
        private final float[][] data;

        public ForFloats(int[] labels, float[][] data) {
            super(labels, data.length);
            this.data = data;
        }

        @Override
        protected void processSubArr(int p, int count, int threadIndex) {
            int[] cardinalities = this.threadCardinalities[threadIndex];
            int[] firstNonZeroIndexesIncreased = this.threadFirstNonZeroIndexesIncreased[threadIndex];
            for (int k = p, kMax = k + count; k < kMax; k++) {
                int label = labels[k];
                if (label > 0) {
                    if (label >= cardinalities.length) {
                        cardinalities = ensureCapacityForLabel(cardinalities, label);
                        firstNonZeroIndexesIncreased = ensureCapacityForLabel(firstNonZeroIndexesIncreased, label);
                    }
                    cardinalities[label]++;
                    if (firstNonZeroIndexesIncreased[label] == 0) {
                        boolean nonZero = false;
                        for (int c = 0; c < numberOfChannels; c++) {
                            if (data[c][k] != 0) {
                                nonZero = true;
                                break;
                            }
                        }
                        if (nonZero) {
                            firstNonZeroIndexesIncreased[label] = k + 1;
                            // - firstNonZeroIndexesIncreased must contain NON-ZERO values to avoid extra time and
                            // efforts for initializing newly allocated arrays by -1 or something like this
                        }
                    }
                }
            }
            this.threadCardinalities[threadIndex] = cardinalities;
            this.threadFirstNonZeroIndexesIncreased[threadIndex] = firstNonZeroIndexesIncreased;
        }

        @Override
        protected void finish() {
            super.finish();
            this.firstNonZeroFloatValues = new float[firstNonZeroIndexes.length * numberOfChannels];
            // - zero-filled by Java
            for (int label = 0, disp = 0; label < firstNonZeroIndexes.length; label++, disp += numberOfChannels) {
                final int index = firstNonZeroIndexes[label];
                if (index >= 0) {
                    for (int c = 0; c < numberOfChannels; c++) {
                        this.firstNonZeroFloatValues[disp + c] = data[c][index];
                    }
                }
            }
        }
    }


    private static class ForDoubles extends FirstNonZeroCalculator {
        private final double[][] data;

        public ForDoubles(int[] labels, double[][] data) {
            super(labels, data.length);
            this.data = data;
        }

        @Override
        protected void processSubArr(int p, int count, int threadIndex) {
            int[] cardinalities = this.threadCardinalities[threadIndex];
            int[] firstNonZeroIndexesIncreased = this.threadFirstNonZeroIndexesIncreased[threadIndex];
            for (int k = p, kMax = k + count; k < kMax; k++) {
                int label = labels[k];
                if (label > 0) {
                    if (label >= cardinalities.length) {
                        cardinalities = ensureCapacityForLabel(cardinalities, label);
                        firstNonZeroIndexesIncreased = ensureCapacityForLabel(firstNonZeroIndexesIncreased, label);
                    }
                    cardinalities[label]++;
                    if (firstNonZeroIndexesIncreased[label] == 0) {
                        boolean nonZero = false;
                        for (int c = 0; c < numberOfChannels; c++) {
                            if (data[c][k] != 0) {
                                nonZero = true;
                                break;
                            }
                        }
                        if (nonZero) {
                            firstNonZeroIndexesIncreased[label] = k + 1;
                            // - firstNonZeroIndexesIncreased must contain NON-ZERO values to avoid extra time and
                            // efforts for initializing newly allocated arrays by -1 or something like this
                        }
                    }
                }
            }
            this.threadCardinalities[threadIndex] = cardinalities;
            this.threadFirstNonZeroIndexesIncreased[threadIndex] = firstNonZeroIndexesIncreased;
        }

        @Override
        protected void finish() {
            super.finish();
            this.firstNonZeroFloatValues = new float[firstNonZeroIndexes.length * numberOfChannels];
            // - zero-filled by Java
            for (int label = 0, disp = 0; label < firstNonZeroIndexes.length; label++, disp += numberOfChannels) {
                final int index = firstNonZeroIndexes[label];
                if (index >= 0) {
                    for (int c = 0; c < numberOfChannels; c++) {
                        this.firstNonZeroFloatValues[disp + c] = (float) data[c][index];
                    }
                }
            }
        }
    }

    /*Repeat.AutoGeneratedEnd*/
}
