/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2017-2023 Daniel Alievsky, AlgART Laboratory (http://algart.net)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

package net.algart.executors.modules.opencv.matrices.filtering;

import net.algart.executors.modules.opencv.common.VoidResultUMatFilter;
import net.algart.executors.modules.util.opencv.OTools;
import net.algart.executors.api.ReadOnlyExecutionInput;
import org.bytedeco.opencv.opencv_core.*;
import org.bytedeco.opencv.global.opencv_imgproc;

public final class CLAHE extends VoidResultUMatFilter implements ReadOnlyExecutionInput {
    private int tileGridSize = 8;
    private double clipLimit = 40.0;
    private boolean autoConvertToGrayscale = false;

    public int getTileGridSize() {
        return tileGridSize;
    }

    public void setTileGridSize(int tileGridSize) {
        this.tileGridSize = positive(tileGridSize);
    }

    public double getClipLimit() {
        return clipLimit;
    }

    public void setClipLimit(double clipLimit) {
        this.clipLimit = clipLimit;
    }

    public boolean isAutoConvertToGrayscale() {
        return autoConvertToGrayscale;
    }

    public void setAutoConvertToGrayscale(boolean autoConvertToGrayscale) {
        this.autoConvertToGrayscale = autoConvertToGrayscale;
    }

    @Override
    public void process(Mat result, Mat source) {
        Mat mat = source;
        try {
            if (autoConvertToGrayscale) {
                mat = OTools.toMonoIfNot(mat);
            }
            try (final Size size = new Size(tileGridSize, tileGridSize)) {
                try (final org.bytedeco.opencv.opencv_imgproc.CLAHE clahe =
                             opencv_imgproc.createCLAHE(clipLimit, size)) {
                    logDebug(() -> "CLAHE: grid tile " + tileGridSize + "x" + tileGridSize
                            + ", clipLimit " + clipLimit + " (source: " + source + ")");
                    clahe.apply(mat, result);
                }
            }
        } finally {
            OTools.closeFirstIfDiffersFromSecond(mat, source);
        }
    }

    @Override
    public void process(UMat result, UMat source) {
        UMat mat = source;
        try {
            if (autoConvertToGrayscale) {
                mat = OTools.toMonoIfNot(mat);
            }
            try (final Size size = new Size(tileGridSize, tileGridSize)) {
                try (final org.bytedeco.opencv.opencv_imgproc.CLAHE clahe =
                             opencv_imgproc.createCLAHE(clipLimit, size)) {
                    logDebug(() -> "CLAHE (GPU): grid tile " + tileGridSize + "x" + tileGridSize
                            + ", clipLimit " + clipLimit + " (source: " + source + ")");
                    clahe.apply(mat, result);
                }
            }
        } finally {
            OTools.closeFirstIfDiffersFromSecond(mat, source);
        }
    }
}