/**
 * Copyright (C) 2018 Dr. David H. Akehurst (http://dr.david.h.akehurst.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.akehurst.language.processor.java8

//import com.soywiz.korio.async.runBlockingNoSuspensions
//import com.soywiz.korio.file.std.resourcesVfs
import net.akehurst.language.agl.processor.Agl
import net.akehurst.language.agl.sppt.SPPT2InputText
import net.akehurst.language.api.parser.ParseFailedException
import net.akehurst.language.api.processor.LanguageProcessor
import kotlin.test.Test
import kotlin.test.assertEquals
import kotlin.test.assertFailsWith
import kotlin.test.assertNotNull

class test_Java8_Singles_aglOptm {

    companion object {
        val grammarFile = "/java8/Java8AglOptm.agl"
        val proc: LanguageProcessor by lazy { createJava8Processor(grammarFile, true) }

        fun createJava8Processor(path: String, toUpper: Boolean = false): LanguageProcessor {
            val grammarStr = this::class.java.getResource(path).readText()
            val proc = Agl.processorFromString(grammarStr)
            val forRule = if (toUpper) "CompilationUnit" else "compilationUnit"
            //proc.buildFor(forRule)//TODO: use build
            return proc
        }
    }

    @Test
    fun literal() {
        val sentence = "0"
        val goal = "Literal"

        val t = proc.parseForGoal(goal, sentence)
    }

    @Test
    fun Types_Type__int() {

        val grammarStr = this::class.java.getResource(grammarFile).readText()
        val goal = "Types.Type"
        val p = Agl.processorFromStringForGoal(grammarStr, goal)

        val sentence = "int"
        val t = p.buildFor(goal).parseForGoal(goal, sentence)//TODO: use build

        assertEquals(1, t.maxNumHeads)
    }

    @Test
    fun Expressions_Type__int() {
        val grammarStr = """
            namespace test

            grammar Expressions {
                leaf ID = "[A-Za-z]"+ ;
                leaf UPT = 'int' ;

                UType = UTypeNonArray Dims? ;
                UTypeNonArray = UTypeReference | UPT ;
                UTypeReference = ID TypeArguments? ;
                TypeArguments = '<' ReferenceType '>' ;
                ReferenceType = Annotation* UTypeReference ;
                Annotation = ID  TypeArguments?  ;
                Dims = Annotation '[' ']' ;
            }
        """.trimIndent()
        val goal = "UType"
        val p = Agl.processorFromStringForGoal(grammarStr, goal)

        val sentence = "int"
        //val t = p.buildFor(goal).parse(goal, sentence)//TODO: use build
        val t = p.parseForGoal(goal, sentence)//TODO: use build

        assertEquals(1, t.maxNumHeads)
    }

    @Test
    fun t1() {
        //val sentence = "import x; @An() interface An {  }"
        val sentence = "import x; @An() interface An {  }"
        val goal = "CompilationUnit"

        val t = proc.parseForGoal(goal, sentence)
    }

    @Test
    fun arrayIndex() {
        val sentence = "a[0]"
        val goal = "Expression"
        val t = proc.parseForGoal(goal, sentence)
    }

    @Test
    fun t() {
        val sentence = "a[0].b"
        val goal = "Expression"
        val t = proc.parseForGoal(goal, sentence)
    }

    @Test
    fun FieldDeclaration() {
        val sentence = "int valid = 0b0;"
        val goal = "FieldDeclaration"
        //proc.parse(goal, sentence)
        proc.buildFor(goal).parseForGoal(goal, sentence)
    }

    @Test
    fun TypeDeclaration_FieldDeclaration() {
        val sentence = "class A { int valid = 0b0; }"
        val goal = "TypeDeclaration"
        proc.buildFor(goal).parseForGoal(goal, sentence)
    }

    @Test
    fun CompilationUnit_FieldDeclaration() {
        val sentence = "class A { int valid = 0b0; }"
        val goal = "CompilationUnit"
        proc.parseForGoal(goal, sentence)
    }

    @Test
    fun bad_Binary_Literal() {
        val sentence = "0b012"
        val goal = "VariableInitializer"
        assertFailsWith(ParseFailedException::class) {
            proc.parseForGoal(goal, sentence)
        }
    }

    @Test
    fun BadLiterals() {
        val sentence = """
/*
 * @test /nodynamiccopyright/
 * @bug 6860965
 * @summary Project Coin: binary literals
 * @compile/fail/ref=BadBinaryLiterals.6.out -XDrawDiagnostics -source 6 -Xlint:-options BadBinaryLiterals.java
 * @compile/fail/ref=BadBinaryLiterals.7.out -XDrawDiagnostics BadBinaryLiterals.java
 */

public class BadBinaryLiterals {
    int valid = 0b0;            // valid literal, illegal in source 6
    int baddigit = 0b011;       // bad digit
                    //aaaabbbbccccddddeeeeffffgggghhhh
    int overflow1 = 0b111111111111111111111111111111111; // too long for int
                    //aaaabbbbccccddddeeeeffffgggghhhhiiiijjjjkkkkllllmmmmnnnnoooopppp
    int overflow2 = 0b11111111111111111111111111111111111111111111111111111111111111111L; // too long for long
    float badfloat1 = 0b01.01;  // no binary floats
    float badfloat2 = 0b01e01;  // no binary floats
}
            """.trimIndent()
        val goal = "CompilationUnit"
        proc.parseForGoal(goal, sentence)

    }

    @Test
    fun UnannQualifiedTypeReference1() {
        val sentence = "Map.Entry<Object,Object> x;"
        val goal = "BlockStatement"
        val t = proc.parseForGoal(goal, sentence)
        assertNotNull(t)
    }

    @Test
    fun UnannQualifiedTypeReference2() {
        val sentence = "Map.Entry<Object,Object> x;"
        val goal = "BlockStatement"
        val t = proc.parseForGoal(goal, sentence)
        assertNotNull(t)
    }

    @Test
    fun UnannQualifiedTypeReference() {
        val sentence = "{ Map.@An Entry<Object,Object> x; }"
        val goal = "Block"
        val t = proc.parseForGoal(goal, sentence)
    }

    @Test
    fun Enum() {
        val sentence = "enum E { A, B, C }"
        val goal = "ClassDeclaration"
        val t = proc.parseForGoal(goal, sentence)
        val actual = t.toStringAll
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun xx() {
        val sentence = "interface An { An[] value(); }"
        val goal = "CompilationUnit"
        val t = proc.parseForGoal(goal, sentence)
        //val actual = t.toStringAll
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun class_with_constructor() {
        val sentence = "class B {  B() {  } }"
        val goal = "CompilationUnit"
        val t = proc.parseForGoal(goal, sentence)
        //val actual = t.toStringAll
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun FormalParameterList_FormalParameter() {
        val sentence = "A a"
        val goal = "FormalParameterList"
        val t = proc.parseForGoal(goal, sentence)
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun FormalParameterList_ReceiverParameter() {
        val sentence = "A this"
        val goal = "FormalParameterList"
        val t = proc.parseForGoal(goal, sentence)
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun FormalParameterList_VarargsParameter() {
        val sentence = "A... this"
        val goal = "FormalParameterList"
        val t = proc.parseForGoal(goal, sentence)
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun FormalParameterList_2() {
        val sentence = "A a, B b"
        val goal = "FormalParameterList"
        val t = proc.parseForGoal(goal, sentence)
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun FormalParameterList_3() {
        val sentence = "A a, B b, C c"
        val goal = "FormalParameterList"
        val t = proc.parseForGoal(goal, sentence)
        //val actual = t.toStringAll
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun ConstructorDeclaration() {
        val sentence = "B() {  }"
        val goal = "ConstructorDeclaration"
        val t = proc.parseForGoal(goal, sentence)
        //val actual = t.toStringAll
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun ConstructorDeclarator() {
        val sentence = "B()"
        val goal = "ConstructorDeclarator"
        val t = proc.parseForGoal(goal, sentence)
        //val actual = t.toStringAll
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test
    fun ConstructorBody() {
        val sentence = "{  }"
        val goal = "ConstructorBody"
        val t = proc.parseForGoal(goal, sentence)
        //val actual = t.toStringAll
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

    @Test//(timeout = 5000)
    fun long_concatenation() {

        val sentence = """
          {
            concat =  "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e"
                    + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" + "a" + "b" + "c" + "d" + "e" ;
          }
        """.trimIndent()
        val goal = "Block"

        val t = proc.parseForGoal(goal, sentence)

        // println( t.toStringAll )
        val resultStr = SPPT2InputText().visitTree(t, "")
        assertEquals(sentence, resultStr)
    }

}