/**
 * Copyright (C) 2018 Dr. David H. Akehurst (http://dr.david.h.akehurst.net)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *         http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.akehurst.language.agl.grammar.grammar

import net.akehurst.language.agl.collections.toSeparatedList
import net.akehurst.language.agl.grammar.grammar.asm.*
import net.akehurst.language.agl.processor.IssueHolder
import net.akehurst.language.agl.syntaxAnalyser.SyntaxAnalyserByMethodRegistrationAbstract
import net.akehurst.language.api.analyser.SyntaxAnalyser
import net.akehurst.language.api.grammar.*
import net.akehurst.language.api.processor.LanguageIssue
import net.akehurst.language.api.processor.LanguageProcessorPhase
import net.akehurst.language.api.processor.SentenceContext
import net.akehurst.language.api.sppt.Sentence
import net.akehurst.language.api.sppt.SpptDataNodeInfo

internal class AglGrammarSyntaxAnalyser(
    //val languageRegistry: LanguageRegistryDefault
) : SyntaxAnalyserByMethodRegistrationAbstract<List<Grammar>>() {

    private val _issues = IssueHolder(LanguageProcessorPhase.SYNTAX_ANALYSIS)

    private val _localStore = mutableMapOf<String, Any>()

    override val embeddedSyntaxAnalyser: Map<String, SyntaxAnalyser<List<Grammar>>> = emptyMap()

    override fun registerHandlers() {
        this.register(this::grammarDefinition)
        this.register(this::namespace)
        this.register(this::definitions)
        this.register(this::grammar)
        this.register(this::options)
        this.register(this::option)
        this.register(this::value)
        this.register(this::extendsOpt)
        this.register(this::extends)
        this.register(this::extendsList)
        this.register(this::rules)
        this.register(this::rule)
        this.register(this::grammarRule)
        this.register(this::overrideRule)
        this.register(this::overrideOperator)
        this.register(this::preferenceRule)
        this.register(this::ruleTypeLabels)
        this.register(this::rhs)
        this.register(this::empty)
        this.register(this::choice)
        this.register(this::simpleChoice)
        this.register(this::priorityChoice)
        this.register(this::ambiguousChoice)
        this.register(this::concatenation)
        this.register(this::concatenationItem)
        this.register(this::simpleItemOrGroup)
        this.register(this::simpleItem)
        this.register(this::listOfItems)
        this.register(this::multiplicity)
        this.register(this::range)
        this.register(this::rangeUnBraced)
        this.register(this::rangeBraced)
        this.register(this::rangeMaxOpt)
        this.register(this::rangeMax)
        this.register(this::rangeMaxBounded)
        this.register(this::rangeMaxUnbounded)
        this.register(this::simpleList)
        this.register(this::group)
        this.register(this::groupedContent)
        this.register(this::separatedList)
        this.register(this::nonTerminal)
        this.register(this::embedded)
        this.register(this::terminal)
        this.register(this::qualifiedName)
        this.register(this::preferenceOption)
        this.register(this::choiceNumber)
        this.register(this::associativity)
    }

    override fun configure(configurationContext: SentenceContext<GrammarItem>, configuration: Map<String, Any>): List<LanguageIssue> {
        //TODO
        return emptyList()
    }

    // grammarDefinition : namespace definitions ;
    private fun grammarDefinition(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<Grammar> {
        val namespace = children[0]
        val definitions = children[1] as List<Grammar>
        return definitions
    }

    // definitions = grammar+ ;
    private fun definitions(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<Grammar> {
        val definitions = children as List<Grammar>
        return definitions
    }

    // namespace : 'namespace' qualifiedName ;
    private fun namespace(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Namespace {
        val qualifiedName = children[1] as String
        val ns = NamespaceDefault(qualifiedName)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
        _localStore["namespace"] = ns
        return ns
    }

    // grammar : 'grammar' IDENTIFIER extendsOpt '{' options rules '}' ;
    private fun grammar(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Grammar {
        val namespace = _localStore["namespace"] as Namespace
        val name = children[1] as String
        val extends = (children[2] as List<GrammarReference>?) ?: emptyList()
        val options = children[4] as List<GrammarOption>
        val rules = children[5] as List<(Grammar) -> GrammarItem>

        val grmr = GrammarDefault(namespace, name, options)
        grmr.extends.addAll(extends)
        _localStore["grammar"] = grmr
        rules.forEach { f ->
            val item = f(grmr)
            when (item) {
                is OverrideRule -> grmr.grammarRule.add(item)
                is GrammarRule -> grmr.grammarRule.add(item)
                is PreferenceRule -> grmr.preferenceRule.add(item)
                else -> error("Not handled")
            }
        }
        return grmr
    }

    // options = option* ;
    private fun options(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<GrammarOption> =
        children as List<GrammarOption>

    // option = '@' IDENTIFIER ':' value ;
    private fun option(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): GrammarOption {
        val name = children[1] as String
        val value = children[3] as String
        return GrammarOptionDefault(name, value)
    }

    // value = IDENTIFIER | LITERAL ;
    private fun value(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): String = when (target.alt.option) {
        0 -> children[0] as String
        1 -> (children[0] as String).let { it.substring(1, it.length - 1) }
        else -> error("Unsupported choice")
    }

    // extendsOpt = extends?
    private fun extendsOpt(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<GrammarReference> {
        return when {
            null == children[0] -> emptyList()
            else -> children[0] as List<GrammarReference>
        }
    }

    // extends = 'extends' extendsList ;
    private fun extends(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<GrammarReference> {
        return children[1] as List<GrammarReference>
    }

    // extendsList = [qualifiedName / ',']+ ;
    private fun extendsList(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<GrammarReference> {
        val localNamespace = _localStore["namespace"] as Namespace
        val extendNameList = children as List<String>
        val sl = extendNameList.toSeparatedList<String, String>()
        val extendedGrammars = sl.items.map {
            // need to manually add the GrammarReference as it is not seen by the super class
            GrammarReferenceDefault(localNamespace, it).also { this.locationMap[it] = sentence.locationFor(target.node) }
        }
        return extendedGrammars
    }

    // rules : rule+ ;
    private fun rules(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<(Grammar) -> GrammarItem> =
        children as List<(Grammar) -> GrammarItem>

    // rule = overrideRule | grammarRule | preferenceRule
    private fun rule(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): (Grammar) -> GrammarItem =
        children[0] as (Grammar) -> GrammarItem

    // grammarRule : ruleTypeLabels IDENTIFIER '=' rhs ';' ;
    private fun grammarRule(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): (Grammar) -> NormalRule {
        val type = children[0] as List<String>
        val isSkip = type.contains("skip")
        val isLeaf = type.contains("leaf")
        val name = children[1] as String
        val rhs = children[3] as RuleItem

        return { grammar ->
            val result = NormalRuleDefault(grammar, name, isSkip, isLeaf)
            result.rhs = rhs
            result
                .also { this.locationMap[it] = sentence.locationFor(target.node) }
        }
    }

    // overrideRule : 'override' ruleTypeLabels IDENTIFIER overrideOperator rhs ';' ;
    private fun overrideRule(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): (Grammar) -> OverrideRule {
        val type = children[1] as List<String>
        val isSkip = type.contains("skip")
        val isLeaf = type.contains("leaf")
        val identifier = children[2] as String
        val overrideOperator = children[3] as String
        val overrideKind = when (overrideOperator) {
            "==" -> OverrideKind.SUBSTITUTION
            "=" -> OverrideKind.REPLACE
            "+=|" -> OverrideKind.APPEND_ALTERNATIVE
            else -> error("overrideOperator $overrideOperator not handled")
        }
        val rhs = children[4] as RuleItem

        return { grammar ->
            val result = OverrideRuleDefault(grammar, identifier, isSkip, isLeaf, overrideKind)
            result.overridenRhs = rhs
            result
                .also { this.locationMap[it] = sentence.locationFor(target.node) }
        }
    }

    // overrideOperator = '=' | '+|' ;
    private fun overrideOperator(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): String =
        children[0] as String

    // ruleTypeLabels : isOverride isSkip isLeaf ;
    // isSkip = 'leaf' ? ;
    // isLeaf = 'skip' ? ;
    private fun ruleTypeLabels(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): List<String> {
        return (children as List<String?>).filterNotNull()
    }

    // rhs = empty | concatenation | choice ;
    private fun rhs(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem {
        val rhs = children[0] as RuleItem
        val rhs2 = when (rhs) {
            is Concatenation -> reduceConcatenation(rhs)
            else -> rhs
        }
        return rhs2
    }

    // empty = ;
    private fun empty(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem {
        return EmptyRuleDefault()//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // choice = ambiguousChoice | priorityChoice | simpleChoice ;
    private fun choice(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem {
        return children[0] as RuleItem
    }

    // simpleChoice : [concatenation, '|']* ;
    private fun simpleChoice(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem {
        val simpleChoice = children.toSeparatedList<Concatenation, String>()
        val alternative = simpleChoice.items.map { reduceConcatenation(it) }
        return ChoiceLongestDefault(alternative)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // priorityChoice : [concatenation, '<']* ;
    private fun priorityChoice(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem {
        val priorityChoice = children.toSeparatedList<Concatenation, String>()
        val alternative = priorityChoice.items.map { reduceConcatenation(it) }
        return ChoicePriorityDefault(alternative)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // ambiguousChoice : [concatenation, '||']* ;
    private fun ambiguousChoice(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem {
        val ambiguousChoice = children.toSeparatedList<Concatenation, String>()
        val alternative = ambiguousChoice.items.map { reduceConcatenation(it) }
        return ChoiceAmbiguousDefault(alternative)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // concatenation : concatenationItem+ ;
    private fun concatenation(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Concatenation {
        val items = children as List<RuleItem>
        return ConcatenationDefault(items)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // concatenationItem = simpleItemOrGroup | listOfItems ; // a group can be mapped to a Choice so return RuleItem
    private fun concatenationItem(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem =
        children[0] as RuleItem

    // simpleItemOrGroup : simpleItem | group ; // a group can be mapped to a Choice so return RuleItem
    private fun simpleItemOrGroup(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem =
        children[0] as RuleItem

    // simpleItem : terminal | nonTerminal | embedded ;
    private fun simpleItem(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): SimpleItem =
        children[0] as SimpleItem

    // listOfItems = simpleList | separatedList ;
    // could also return optional
    private fun listOfItems(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): ConcatenationItem =
        children[0] as ConcatenationItem

    // multiplicity = '*' | '+' | '?' | range ;
    private fun multiplicity(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Pair<Int, Int> {
        val m = children[0]
        return when (m) {
            is String -> when (m) {
                "*" -> Pair(0, -1)
                "+" -> Pair(1, -1)
                "?" -> Pair(0, 1)
                else -> error("should not happen")
            }

            is Pair<*, *> -> m as Pair<Int, Int>
            else -> error("should not happen")
        }
    }

    //range = rangeBraced | rangeUnBraced ;
    private fun range(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Pair<Int, Int> =
        children[0] as Pair<Int, Int>

    //rangeUnBraced = POSITIVE_INTEGER rangeMaxOpt ;
    private fun rangeUnBraced(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Pair<Int, Int> {
        val min = (children[0] as String).toInt()
        val max = if (children[1] == null) {
            min
        } else {
            children[1] as Int
        }
        return Pair(min, max)
    }

    //rangeBraced = '{' POSITIVE_INTEGER rangeMaxOpt '}' ;
    private fun rangeBraced(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Pair<Int, Int> {
        val min = (children[1] as String).toInt()
        val max = if (null == children[2]) {
            min
        } else {
            children[2] as Int
        }
        return Pair(min, max)
    }

    // rangeMaxOpt = rangeMax? ;
    private fun rangeMaxOpt(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Int? {
        return children[0] as Int?
    }

    //rangeMax = rangeMaxUnbounded | rangeMaxBounded ;
    private fun rangeMax(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Int =
        children[0] as Int


    //rangeMaxUnbounded = '+' ;
    private fun rangeMaxUnbounded(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Int = -1

    //rangeMaxBounded = '..' POSITIVE_INTEGER ;
    private fun rangeMaxBounded(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Int =
        (children[1] as String).toInt()

    // simpleList = simpleItem multiplicity ;
    private fun simpleList(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): ConcatenationItem {
        val (min, max) = children[1] as Pair<Int, Int>
        val item = children[0] as RuleItem
        return when {
            min == 0 && max == 1 -> OptionalItemDefault(item)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
            else -> SimpleListDefault(min, max, item)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
        }
    }

    // separatedList : '[' simpleItem '/' terminal ']' multiplicity ;
    private fun separatedList(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): SeparatedList {
        val (min, max) = children[5] as Pair<Int, Int>
        val separator = children[3] as RuleItem
        val item = children[1] as RuleItem
        return SeparatedListDefault(min, max, item, separator)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // group = '(' groupedContent ')' ;
    private fun group(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem {
        val groupContent = children[1] as RuleItem
        return when (groupContent) {
            is Choice -> groupContent//.also { this.locationMap[it] = target.node.locationIn(sentence) }
            is Concatenation -> {
                val reduced = reduceConcatenation(groupContent)
                GroupDefault(reduced)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
            }

            else -> error("Internal Error: subtype of RuleItem not handled - ${groupContent::class.simpleName}")
        }
    }

    // groupedContent = concatenation | choice ;
    private fun groupedContent(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): RuleItem =
        children[0] as RuleItem


    // nonTerminal : qualifiedName ;
    private fun nonTerminal(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): NonTerminal {
        val qualifiedName = children[0] as String
        return if (qualifiedName.contains(".")) {
            val localNamespace = _localStore["namespace"] as Namespace
            val nonTerminalQualified = children[0] as String
            val nonTerminalRef = nonTerminalQualified.substringAfterLast(".")
            val grammarRef = nonTerminalQualified.substringBeforeLast(".")
            val gr = GrammarReferenceDefault(localNamespace, grammarRef).also { this.locationMap[it] = sentence.locationFor(target.node) }
            val nt = NonTerminalDefault(gr, nonTerminalRef)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
            return nt
        } else {
            val nt = NonTerminalDefault(null, qualifiedName)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
            nt
        }
    }

    // embedded = qualifiedName '::' nonTerminal ;
    private fun embedded(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Embedded {
        val namespace = _localStore["namespace"] as Namespace
        val embeddedGrammarStr = children[0] as String
        val embeddedStartRuleRef = children[2] as NonTerminal
        val embeddedGrammarRef = GrammarReferenceDefault(namespace, embeddedGrammarStr)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
        return EmbeddedDefault(embeddedStartRuleRef.name, embeddedGrammarRef)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // terminal : LITERAL | PATTERN ;
    private fun terminal(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Terminal {
        // Must match what is done in AglStyleSyntaxAnalyser.selectorSingle
        val isPattern = when (target.alt.option) {
            0 -> false
            else -> true
        }
        val mt = children[0] as String
        val escaped = mt.substring(1, mt.length - 1)
        //TODO: check these unescapings, e.g. '\\n'
//        val value = if (isPattern) {
//            escaped.replace("\\\"", "\"")
//        } else {
//            escaped.replace("\\'", "'").replace("\\\\", "\\")
//        }
        return TerminalDefault(escaped, isPattern)//.also { this.locationMap[it] = target.node.locationIn(sentence) }
    }

    // qualifiedName : (IDENTIFIER / '.')+ ;
    private fun qualifiedName(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): String {
        return (children as List<String>).joinToString(separator = "")
    }

    // preferenceRule = 'preference' simpleItem '{' preferenceOptionList '}' ;
    private fun preferenceRule(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): (Grammar) -> PreferenceRule {
        val forItem = children[1] as SimpleItem
        val optionList = children[3] as List<PreferenceOption>
        return { grammar ->
            PreferenceRuleDefault(grammar, forItem, optionList)
                .also { this.locationMap[it] = sentence.locationFor(target.node) }
        }
    }

    // preferenceOption = nonTerminal choiceNumber 'on' terminalList associativity ;
    private fun preferenceOption(target: SpptDataNodeInfo, children: List<Any?>, arg: Any?): PreferenceOption {
        val item = children[0] as NonTerminal
        val choiceNumber = when {
            null == children[1] -> 0
            children[1] is String -> (children[1] as String).toInt() //FIXME: choiceNumber not called!
            else -> children[1] as Int
        }
        val terminalList = children[3] as List<SimpleItem>
        val assStr = children[4] as String
        val associativity = when (assStr) {
            "left" -> PreferenceOption.Associativity.LEFT
            "right" -> PreferenceOption.Associativity.RIGHT
            else -> error("Internal Error: associativity value '$assStr' not supported")
        }
        return PreferenceOptionDefault(item, choiceNumber, terminalList, associativity)
    }

    // choiceNumber = POSITIVE_INTEGER? ;
    private fun choiceNumber(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): Int? {
        val n = children[0] as String?
        return n?.toInt()
    }

    // associativity = 'left' | 'right' ;
    private fun associativity(target: SpptDataNodeInfo, children: List<Any?>, sentence: Sentence): String =
        children[0] as String

    private fun reduceConcatenation(concat: Concatenation): RuleItem {
        return when (concat.items.size) {
            1 -> when (concat.items[0]) {
                is TangibleItem -> concat.items[0]
                is OptionalItem -> concat.items[0]
                is ListOfItems -> concat.items[0]
                is Choice -> concat.items[0]
                is Group -> {
                    val content = (concat.items[0] as Group).groupedContent
                    when (content) {
                        is Choice -> content
                        else -> concat
                    }
                }

                else -> error("Internal Error: subtype of ConcatenationItem not handled - ${concat.items[0]::class.simpleName}")
            }

            else -> concat
        }
    }

}