package net.aihelp.utils;

import android.content.Context;
import android.text.TextUtils;

import net.aihelp.BuildConfig;
import net.aihelp.config.AIHelpContext;
import net.aihelp.data.localize.data.LocaleStringHelper;

public final class ResResolver {

    // Assumes input string is in the format "foo.bar.baz" and strips out the last component.
    // Returns null on failure.
    private static String removeOneSuffix(String input) {
        if (input == null) return null;
        int lastDotIndex = input.lastIndexOf('.');
        if (lastDotIndex == -1) return null;
        return input.substring(0, lastDotIndex);
    }

    private static Class<?> loadClazz(String packageName, String assetType) throws ClassNotFoundException {
        Context context = AIHelpContext.getInstance().getContext();
        return context.getClassLoader().loadClass(packageName + ".R$" + assetType);
    }

    /**
     * If getIdentifier API failed, we will try to reflect R.java to get resources during runtime.
     * <p>
     * By default, all resource will be merged into base.apk, but the tricky part is R.java will be merged
     * into package's root path where you're about to retrieve resource ids from, but the return of Context#getPackageName()
     * is the applicationId configured in build.gradle.
     * <p>
     * To make things right, we will forget about the host application, and get resource ids directly from AIHelp package.
     * <p>
     * However, when the host application is using Google aab dynamic delivery, after merging aihelp.apk into base.apk,
     * we properly can't found AIHelp package at all, and ClassNotFoundException would be thrown.
     *
     * In this Condition, we will try to reflect host application's package to get what we want.
     * But the Context#packageName returns gradle applicationId, which could be different from the real packageName.
     * If we can't get an valid class instance from applicationId, then we will remove one suffix and try again until we
     * found the right packageName.
     *
     * Note: This method could be failed if the applicationId is totally different from the real packageName,
     * for example, use foo.bar.baz as packageName and make bar.baz.qux as applicationId.
     * In this condition, the AIHelp UI will be messed up, but no crashes should be happening.
     *
     */
    private static Class<?> getCorrectResourceClass(String assetType) throws ClassNotFoundException {
        Class<?> clazz = null;
        try {
            clazz = loadClazz(BuildConfig.LIBRARY_PACKAGE_NAME, assetType);
        } catch (ClassNotFoundException ex) {
            String packageName = AIHelpContext.getInstance().getContext().getPackageName();
            try {
                clazz = loadClazz(packageName, assetType);
            } catch (ClassNotFoundException e) {
                while (clazz == null) {
                    packageName = removeOneSuffix(packageName);
                    if (packageName == null) throw e;
                    try {
                        clazz = loadClazz(packageName, assetType);
                    } catch (ClassNotFoundException ignored) {
                        // ignored this exception, remove one suffix and continue the loop then
                    }
                }
            }
        }
        return clazz;
    }


    public static int[] getStyleable(String name) {
        try {
            return (int[]) getCorrectResourceClass("styleable").getField(name).get(null);
        } catch (Throwable ignored) {
            return null;
        }
    }

    public static int getStyleableFieldIndex(String styleableName, String styleableFieldName) {
        try {
            String name = styleableName + "_" + styleableFieldName;
            return getCorrectResourceClass("styleable").getField(name).getInt(null);
        } catch (Throwable ignored) {
            return 0;
        }
    }

    public static int getViewId(String name) {
        return getIdentifier(name, "id");
    }

    public static int getLayoutId(String name) {
        return getIdentifier(name, "layout");
    }

    public static int getAnimId(String name) {
        return getIdentifier(name, "anim");
    }

    public static int getColorId(String name) {
        return getIdentifier(name, "color");
    }

    public static int getStringId(String name) {
        return getIdentifier(name, "string");
    }

    public static String getString(String name) {
        String string = LocaleStringHelper.INSTANCE.getString(name);
        if (!TextUtils.isEmpty(string)) {
            return string;
        }
        Context context = AIHelpContext.getInstance().getContext();
        return context.getResources().getString(getStringId(name));
    }

    public static int getStyleId(String name) {
        return getIdentifier(name, "style");
    }

    public static int getAttrId(String name) {
        return getIdentifier(name, "attr");
    }

    public static int getDrawableId(String name) {
        return getIdentifier(name, "drawable");
    }

    /**
     * Retrieve app resource ids during runtime.
     *
     * Rely on getResources().getIdentifier for almost all cases, even the packageName is different from applicationId,
     * but getIdentifier will always return 0 for custom styleable, so we will fallback to reflection in this case.
     *
     * Reflection relies on packageName to get resources classes, but the Android API will never return
     * the right packageName for us, we also need to do some optimization for this.
     *
     * Check {@link #getCorrectResourceClass(String) for more infomation}.
     */
    public static int getIdentifier(String name, String assetType) {
        Context context = AIHelpContext.getInstance().getContext();
        int id = context.getResources().getIdentifier(name, assetType, context.getPackageName());
        if (id != 0) {
            return id;
        }
        // Resource id can't be found, using Resources class so fallback to reflection.
        try {
            return getCorrectResourceClass(assetType).getField(name).getInt(null);
        } catch (Exception ignored) {
            return 0;
        }
    }

}