package net.aihelp.ui.widget;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.os.Build;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.View;
import android.widget.TextView;

import net.aihelp.utils.Styles;

import java.util.ArrayList;
import java.util.List;

import androidx.core.view.ViewCompat;

public class StepView extends View {

    private Paint bgPaint;
    private Paint progressPaint;

    private final float radius = 12;

    private String backgroundColor = "#d8d8d8";
    private String progressColor = "#ffffff";
    private int textSize = 14;

    private float startX;
    private float stopX;
    private float bgCenterY;

    private int maxStep = 5;
    private int currentStep = 1;

    private int textPadding = 15;
    private int interval;

    private List<String> steps = new ArrayList<>();

    public boolean isFinished() {
        return currentStep >= maxStep;
    }

    public void setBackgroundColor(String backgroundColor) {
        this.backgroundColor = backgroundColor;
    }

    public void update(List<String> steps, String currentStep, String progressColor) {
        this.steps = steps;
        this.maxStep = steps.size();
        this.currentStep = steps.indexOf(currentStep) + 1;
        this.progressColor = progressColor;
        progressPaint.setColor(Color.parseColor(progressColor));
        invalidate();
    }

    public StepView(Context context) {
        this(context, null);
    }

    public StepView(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public StepView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        textSize = dpToPx(textSize);
        textPadding = dpToPx(textPadding);

        bgPaint = new Paint();
        bgPaint.setAntiAlias(true);
        bgPaint.setStyle(Paint.Style.FILL);
        bgPaint.setColor(Color.parseColor(backgroundColor));
        bgPaint.setStrokeWidth(2f);
        bgPaint.setTextSize(textSize);
        bgPaint.setTextAlign(Paint.Align.CENTER);

        progressPaint = new Paint();
        progressPaint.setAntiAlias(true);
        progressPaint.setStyle(Paint.Style.FILL);
        progressPaint.setColor(Color.parseColor(progressColor));
        progressPaint.setStrokeWidth(2f);
        progressPaint.setTextSize(textSize);
        progressPaint.setTextAlign(Paint.Align.CENTER);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        int widthMode = MeasureSpec.getMode(widthMeasureSpec);
        int heightMode = MeasureSpec.getMode(heightMeasureSpec);

        int bgWidth;
        if (widthMode == MeasureSpec.EXACTLY) {
            bgWidth = MeasureSpec.getSize(widthMeasureSpec);
        } else {
            bgWidth = getPaddingLeft() + dpToPx(300) + getPaddingRight();
        }

        int bgHeight;
        if (heightMode == MeasureSpec.EXACTLY) {
            bgHeight = MeasureSpec.getSize(heightMeasureSpec) - getPaddingTop() - getPaddingBottom();
        } else {
            bgHeight = (int) (textPadding * 2 + textSize + radius * 2) + getPaddingTop() + getPaddingBottom();
        }

        startX = getPaddingLeft() + radius;
        stopX = bgWidth - radius - getPaddingRight();

        if (Styles.isLayoutRtl(this)) {
            startX = bgWidth - getPaddingLeft() - radius;
            stopX = radius + getPaddingRight();
        }

        bgCenterY = bgHeight * 1.0f / 2;
        interval = (int) ((stopX - startX) / (maxStep - 1));

        super.onMeasure(MeasureSpec.makeMeasureSpec(bgWidth, widthMode), MeasureSpec.makeMeasureSpec(bgHeight, heightMode));
    }

    @Override
    protected void onDraw(Canvas canvas) {
        drawBg(canvas);
        drawProgress(canvas);
        drawText(canvas);
    }

    public float getBaseLine(float y, Paint mPaint) {
        Paint.FontMetrics fontMetrics = mPaint.getFontMetrics();
        float distance = (fontMetrics.bottom - fontMetrics.top) / 2 - fontMetrics.bottom;
        return y + distance;
    }

    private void drawText(Canvas canvas) {
        for (int i = 0; i < maxStep; i++) {
            int y = (int) (i % 2 == 0 ? (getBaseLine(bgCenterY, progressPaint) - textPadding) : (getBaseLine(bgCenterY, progressPaint) + textPadding));
            if (i < currentStep) {
                if (null != steps && i < steps.size()) {
                    float x = getDrawTextStartPosition(i, progressPaint);
                    canvas.drawText(steps.get(i), x, y, progressPaint);
                }
            } else {
                if (null != steps && i < steps.size()) {
                    float x = getDrawTextStartPosition(i, bgPaint);
                    canvas.drawText(steps.get(i), x, y, bgPaint);
                }
            }
        }
    }

    private float getDrawTextStartPosition(int index, Paint paint) {
        float x = startX + (index * interval);
        if (index == 0) {
            if (Styles.isLayoutRtl(this)) {
                paint.setTextAlign(Paint.Align.RIGHT);
                x = x + radius;
            } else {
                paint.setTextAlign(Paint.Align.LEFT);
                x = x - radius;
            }
        } else if (index == maxStep - 1) {
            if (Styles.isLayoutRtl(this)) {
                paint.setTextAlign(Paint.Align.LEFT);
                x = x - radius;
            } else {
                paint.setTextAlign(Paint.Align.RIGHT);
                x = x + radius;
            }
        } else {
            paint.setTextAlign(Paint.Align.CENTER);
        }
        return x;
    }

    private void drawProgress(Canvas canvas) {
        int linePro;
        float lastLeft = startX;
        for (int i = 0; i < currentStep; i++) {
            if (i == 0) {
                linePro = 0;
            } else {
                linePro = interval;
            }
            canvas.drawLine(lastLeft, bgCenterY, lastLeft + linePro, bgCenterY, progressPaint);
            lastLeft = lastLeft + linePro;
            canvas.drawCircle(startX + (i * interval), bgCenterY, radius, progressPaint);
        }
    }

    private void drawBg(Canvas canvas) {
        canvas.drawLine(startX, bgCenterY, stopX, bgCenterY, bgPaint);
        for (int i = 0; i < maxStep; i++) {
            canvas.drawCircle(startX + (i * interval), bgCenterY, radius, bgPaint);
        }
    }

    public int dpToPx(float dp) {
        return (int) (dp * getContext().getResources().getDisplayMetrics().density);
    }

    private boolean isLayoutRtl() {
        return ViewCompat.getLayoutDirection(this) == ViewCompat.LAYOUT_DIRECTION_RTL;
    }

}
