package net.aihelp.ui.widget;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.drawable.Drawable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.MotionEvent;
import android.view.View;


import net.aihelp.common.CustomConfig;
import net.aihelp.ui.helper.BitmapHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;
import net.aihelp.utils.TLog;

import androidx.annotation.Nullable;
import androidx.core.graphics.drawable.DrawableCompat;
import androidx.core.view.ViewCompat;
import androidx.vectordrawable.graphics.drawable.VectorDrawableCompat;

public class AIHelpRatingBar extends View {

    private Bitmap mFocusedDrawable, mUnfocusedDrawable;

    private int mSpacing, mGradeLevel, mCurrGrade;

    private int mStarWidth = 30, mStarHeight = 30;

    public int getSelectGrade() {
        return mCurrGrade;
    }

    public boolean isFullStar() {
        return mCurrGrade == mGradeLevel;
    }

    public AIHelpRatingBar(Context context) {
        this(context, null);
    }

    public AIHelpRatingBar(Context context, @Nullable AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public AIHelpRatingBar(Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        int[] styleable = ResResolver.getStyleable("aihelp_rating_bar");
        if (styleable != null) {
            TypedArray typedArray = context.obtainStyledAttributes(attrs, styleable);

            int focusedResourceId = typedArray.getResourceId(ResResolver.getStyleableFieldIndex(
                            "aihelp_rating_bar", "aihelp_rating_bar_focused"),
                    ResResolver.getDrawableId("aihelp_svg_star_selected"));
            int unfocusedResourceId = typedArray.getResourceId(ResResolver.getStyleableFieldIndex(
                            "aihelp_rating_bar", "aihelp_rating_bar_unfocused"),
                    ResResolver.getDrawableId("aihelp_svg_star_unselected"));
            mSpacing = typedArray.getDimensionPixelSize(ResResolver.getStyleableFieldIndex(
                    "aihelp_rating_bar", "aihelp_rating_bar_horizontal_spacing"), dip2px(context, 27));
            mGradeLevel = typedArray.getInt(ResResolver.getStyleableFieldIndex(
                    "aihelp_rating_bar", "aihelp_rating_bar_grade_level"), 5);
            mCurrGrade = typedArray.getInt(ResResolver.getStyleableFieldIndex(
                    "aihelp_rating_bar", "aihelp_rating_bar_default_grade"), mGradeLevel);

            typedArray.recycle();

            // 获取星星的 Bitmap 对象
            if (focusedResourceId != 0) {
                mFocusedDrawable = getDrawableBitmap(context, focusedResourceId, Styles.getColor(CustomConfig.CommonSetting.highlightedColor));
            }
            if (unfocusedResourceId != 0) {
                mUnfocusedDrawable = getDrawableBitmap(context, unfocusedResourceId,
                        Styles.getColorWithAlpha(CustomConfig.CommonSetting.highlightedColor, 0.2f));
            }

            if (mFocusedDrawable != null) {
                mStarWidth = mFocusedDrawable.getWidth();
                mStarHeight = mFocusedDrawable.getHeight();
            }
        }
    }

    protected int dip2px(Context context, double dpValue) {
        if (context == null) return (int) dpValue;
        float density = context.getResources().getDisplayMetrics().density;
        return (int) (dpValue * density + 0.5);
    }

    @Override
    protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec);

        // 宽应该是 一个星星的宽度 * 星星的数量；高应该是 星星的高度 即可
        int width = mStarWidth * mGradeLevel + mSpacing * (mGradeLevel - 1);
        int height = mStarHeight;

        setMeasuredDimension(width, height);

    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        if (mFocusedDrawable == null || mUnfocusedDrawable == null) return;

        if (isLayoutRtl()) {
            // 如果是RTL布局，将画布翻转并平移
            canvas.translate(getWidth(), 0);
            canvas.scale(-1, 1);
        }

        for (int i = 0; i < mGradeLevel; i++) {

            int x = mStarWidth * i;
            if (i > 0) {
                x = mStarWidth * i + mSpacing * i;
            }

            if (mCurrGrade > i) {
                // 假设分数为 2 分，0 和 1 需要绘制 mFocusedDrawable，>1 的部分绘制 mUnfocusedDrawable
                canvas.drawBitmap(mFocusedDrawable, x, 0, null);
            } else {
                canvas.drawBitmap(mUnfocusedDrawable, x, 0, null);
            }

        }


    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {

        if (mFocusedDrawable == null || mUnfocusedDrawable == null) return true;

        if (isLayoutRtl()) {
            // 如果是RTL布局，将X坐标翻转
            event.setLocation(getWidth() - event.getX(), event.getY());
        }

        switch (event.getAction()) {
            case MotionEvent.ACTION_MOVE:
            case MotionEvent.ACTION_UP:
                // 修正 x 的值，有可能为负值
                float eventX = event.getX();
                if (eventX < 0) {
                    eventX = 0;
                }

                // 获取触摸位置的 x 坐标，与单个星星的宽度做除法，得到分数值
                // 除完以后需要有个 +1 的矫正操作
                int currGrade = (int) (eventX / (mFocusedDrawable.getWidth() + mSpacing)) + 1;

                // 屏蔽非法值
                if (currGrade < 0) {
                    currGrade = 1;
                } else if (currGrade > mGradeLevel) {
                    currGrade = mGradeLevel;
                }

                // 分数如果相同，则不再进行重绘，避免 onDraw()方法的多次调用
                if (mCurrGrade == currGrade) {
                    return true;
                }

                // 更新变量
                mCurrGrade = currGrade;

                if (mListener != null) {
                    mListener.onRateStatusChanged(mCurrGrade);
                }

                // 更新当前分数后重新绘制页面
                invalidate();

        }

        // 自己消费触摸事件
        return true;
    }

    public static Bitmap getDrawableBitmap(final Context context, int resId, int color) {
        final Bitmap bitmap;
//        Drawable drawable = AppCompatResources.getDrawable(context, resId);
        VectorDrawableCompat drawable = VectorDrawableCompat.create(context.getResources(), resId, null);
        if (drawable != null) {
            DrawableCompat.setTint(DrawableCompat.wrap(drawable).mutate(), color);
        }
        bitmap = Bitmap.createBitmap(drawable.getIntrinsicWidth(),
                drawable.getIntrinsicHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        drawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
        drawable.draw(canvas);
        return bitmap;
    }

    private boolean isLayoutRtl() {
        return ViewCompat.getLayoutDirection(this) == ViewCompat.LAYOUT_DIRECTION_RTL;
    }

    public interface OnStatusChangedListener {
        void onRateStatusChanged(int newRateValue);
    }

    private OnStatusChangedListener mListener;

    public void setOnStatusChangedListener(OnStatusChangedListener mListener) {
        this.mListener = mListener;
    }

}
