package net.aihelp.ui.widget;

import android.animation.LayoutTransition;
import android.app.Activity;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Bitmap;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import net.aihelp.core.ui.image.Picasso;
import net.aihelp.core.ui.image.TargetAdapter;
import net.aihelp.core.ui.loading.indicator.LoadingIndicatorView;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.data.model.task.ReplyMessage;
import net.aihelp.utils.DomainSupportHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.content.res.AppCompatResources;
import androidx.appcompat.widget.AppCompatImageView;

public class AIHelpLoadingMediaView extends FrameLayout {

    private AppCompatImageView imageView;
    private AppCompatImageView ivPlay;
    private View maskView;
    private LoadingIndicatorView loadingView;

    private boolean isVideo;

    private boolean isLoading;

    public boolean isLoading() {
        return isLoading;
    }

    public void setLoading(boolean loading) {
        isLoading = loading;
    }

    public AppCompatImageView getRealImageView() {
        return imageView;
    }

    public AIHelpLoadingMediaView(@NonNull Context context) {
        super(context);
    }

    public AIHelpLoadingMediaView(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        getAttributes(context, attrs);
        init(context);
    }

    public AIHelpLoadingMediaView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        getAttributes(context, attrs);
        init(context);
    }

    private void getAttributes(Context context, AttributeSet attrs) {
        int[] styleable = ResResolver.getStyleable("aihelp_widget");
        if (styleable != null) {
            TypedArray ta = context.getTheme().obtainStyledAttributes(attrs, styleable, 0, 0);
            isVideo = ta.getBoolean(ResResolver.getStyleableFieldIndex("aihelp_widget",
                    "aihelp_widget_is_video"), false);
            ta.recycle();
        }
    }

    private void init(Context context) {

        setLayoutTransition(new LayoutTransition());
        setBackground(AppCompatResources.getDrawable(context, ResResolver.getDrawableId("aihelp_bg_uploading_mask")));
        setForegroundGravity(Gravity.CENTER);

        View rootView = inflate(context, ResResolver.getLayoutId("aihelp_loading_image_view"), this);
        imageView = rootView.findViewById(ResResolver.getViewId("aihelp_image_view"));
        ivPlay = rootView.findViewById(ResResolver.getViewId("aihelp_iv_play"));
        maskView = rootView.findViewById(ResResolver.getViewId("aihelp_v_mask"));
        loadingView = rootView.findViewById(ResResolver.getViewId("aihelp_loading_view"));
        updateLoadingStatus(true);
    }

    public void resetStatus() {
        ViewGroup.LayoutParams layoutParams = getLayoutParams();
        layoutParams.width = Styles.dpToPx(getContext(), 100);
        layoutParams.height = Styles.dpToPx(getContext(), 150);
        setLayoutParams(layoutParams);
        updateLoadingStatus(true);
    }

    public void updateLoadingStatus(boolean showLoading) {
        maskView.setVisibility(showLoading ? VISIBLE : GONE);
        loadingView.setVisibility(showLoading ? VISIBLE : GONE);
        if (isVideo) {
            ivPlay.setVisibility(showLoading ? GONE : VISIBLE);
        }
        setLoading(showLoading);
    }

    public void loadIntoImageView(final Context context, final ConversationMsg conversationMsg) {
        if (context instanceof Activity && !((Activity) context).isFinishing()) {
            final String loadSource = DomainSupportHelper.getAdjustedUrl(
                    isVideo ? conversationMsg.getVideoThumbnail() : conversationMsg.getMsgContent()
            );

            // When user manually send images or videos, the message content will always be the image's or
            // the video thumbnail's absolute path.
            // In this case, the loading status should be reset by the msg SUCCESS status.
            if (conversationMsg.getMsgStatus() == ConversationMsg.STATUS_SUCCESS) {
                if (isVideo || !conversationMsg.isMessageFromServer()) {
                    updateLoadingStatus(false);
                }
            }

            if (conversationMsg.getImageSize() != null) {
                Picasso.get().load(loadSource).fit().into(imageView);
                ViewGroup.LayoutParams rootLayoutParams = AIHelpLoadingMediaView.this.getLayoutParams();
                ViewGroup.LayoutParams params = imageView.getLayoutParams();
                rootLayoutParams.width = params.width = conversationMsg.getImageSize()[0];
                rootLayoutParams.height = params.height = conversationMsg.getImageSize()[1];
                setLayoutParams(rootLayoutParams);
                imageView.setLayoutParams(params);
                updateLoadingStatus(false);
                return;
            }

            resetStatus();
            Picasso.get().load(loadSource).into(new TargetAdapter() {
                @Override
                public void onBitmapLoaded(Bitmap resource, Picasso.LoadedFrom from) {
                    if (conversationMsg.getImageSize() == null) {
                        int[] size = new int[2];
                        ViewGroup.LayoutParams params = AIHelpLoadingMediaView.this.getLayoutParams();

                        float scale = Math.max(resource.getWidth(), resource.getHeight()) / 700f;
                        if (scale < 1) scale = 1;

                        size[0] = params.width = Math.max(resource.getWidth() / (int) Math.ceil(scale), dip2px(getContext(), 50));
                        size[1] = params.height = Math.max(resource.getHeight() / (int) Math.ceil(scale), dip2px(getContext(), 50));

                        conversationMsg.setImageSize(size);
                        AIHelpLoadingMediaView.this.setLayoutParams(params);
                    }

                    Picasso.get().load(loadSource).fit().into(imageView);

                    // As for the left cases, such as received images, history messages, etc,
                    // the msg content will always be http urls, just reset the loading status after the #into() finished.
                    updateLoadingStatus(false);
                }
            });

        }
    }

    public void loadIntoImageView(final Context context, final ReplyMessage replyMessage) {
        if (context instanceof Activity && !((Activity) context).isFinishing()) {
            final String loadSource = DomainSupportHelper.getAdjustedUrl(
                    isVideo ? replyMessage.getVideoThumbnail() : replyMessage.getContent()
            );

            // When user manually send images or videos, the message content will always be the image's or
            // the video thumbnail's absolute path.
            // In this case, the loading status should be reset by the msg SUCCESS status.
            if (replyMessage.getMsgStatus() == ConversationMsg.STATUS_SUCCESS) {
                if (isVideo) {
                    updateLoadingStatus(false);
                }
            }

            if (replyMessage.getImageSize() != null) {
                Picasso.get().load(loadSource).fit().into(imageView);
                ViewGroup.LayoutParams rootLayoutParams = AIHelpLoadingMediaView.this.getLayoutParams();
                ViewGroup.LayoutParams params = imageView.getLayoutParams();
                rootLayoutParams.width = params.width = replyMessage.getImageSize()[0];
                rootLayoutParams.height = params.height = replyMessage.getImageSize()[1];
                setLayoutParams(rootLayoutParams);
                imageView.setLayoutParams(params);
                updateLoadingStatus(false);
                return;
            }

            resetStatus();
            Picasso.get().load(loadSource).into(new TargetAdapter() {
                @Override
                public void onBitmapLoaded(Bitmap resource, Picasso.LoadedFrom from) {
                    if (replyMessage.getImageSize() == null) {
                        int[] size = new int[2];
                        ViewGroup.LayoutParams params = AIHelpLoadingMediaView.this.getLayoutParams();

                        float scale = Math.max(resource.getWidth(), resource.getHeight()) / 700f;
                        if (scale < 1) scale = 1;

                        size[0] = params.width = Math.max(resource.getWidth() / (int) Math.ceil(scale), dip2px(getContext(), 50));
                        size[1] = params.height = Math.max(resource.getHeight() / (int) Math.ceil(scale), dip2px(getContext(), 50));

                        replyMessage.setImageSize(size);
                        AIHelpLoadingMediaView.this.setLayoutParams(params);
                    }

                    Picasso.get().load(loadSource).fit().into(imageView);

                    // As for the left cases, such as received images, history messages, etc,
                    // the msg content will always be http urls, just reset the loading status after the #into() finished.
                    updateLoadingStatus(false);
                }
            });

        }
    }

    public int dip2px(Context context, double dpValue) {
        float density = context.getResources().getDisplayMetrics().density;
        return (int) (dpValue * density + 0.5);
    }

    public void showPlayButton(boolean show) {
        ivPlay.setVisibility(show ? VISIBLE : GONE);
    }

}
