package net.aihelp.ui.task.detail;

import android.Manifest;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.view.WindowManager;
import android.widget.EditText;
import android.widget.ImageView;

import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.SpKeys;
import net.aihelp.core.ui.dialog.AlertDialog;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.luban.Luban;
import net.aihelp.core.util.luban.OnCompressListener;
import net.aihelp.core.util.permission.AIHelpPermissions;
import net.aihelp.core.util.permission.IPermissionCallback;
import net.aihelp.core.util.permission.Permission;
import net.aihelp.data.model.task.ReplyMessage;
import net.aihelp.ui.helper.AttachmentPicker;
import net.aihelp.ui.helper.AttachmentPickerFile;
import net.aihelp.ui.widget.snackbar.Snackbar;
import net.aihelp.ui.wrapper.TextWatcherWrapper;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.RegexDefinition;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.Styles;
import net.aihelp.utils.ToastUtil;
import net.aihelp.utils.UploadFileHelper;

import java.io.File;
import java.util.regex.Pattern;

import androidx.appcompat.widget.AppCompatImageButton;
import androidx.fragment.app.Fragment;

public class TaskReplyComposer implements View.OnClickListener, AttachmentPicker.AttachmentPickerListener, DialogInterface.OnDismissListener {

    private Fragment fragment;
    private Context context;

    private AlertDialog dialog;
    private EditText etInput;
    private AppCompatImageButton btnSend;

    private OnUserReplyListener onUserReplyListener;

    public void setOnUserReplyListener(OnUserReplyListener onUserReplyListener) {
        this.onUserReplyListener = onUserReplyListener;
    }

    public void showReplyComposer(TaskDetailFragment fragment, Context context) {
        this.fragment = fragment;
        this.context = context;

        dialog = new AlertDialog.Builder(context).setContentView(ResResolver.getLayoutId("aihelp_ada_reply_ongoing_task")).fromBottom(true).setCancelableOntheOutside(true).fullWidth().create();

        View container = dialog.findViewById(ResResolver.getViewId("aihelp_ll_input"));
        container.setBackgroundColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha));

        ImageView ivAttach = dialog.findViewById(ResResolver.getViewId("aihelp_iv_attach"));
        ivAttach.setOnClickListener(this);
        Styles.reRenderImageView(ivAttach, "aihelp_svg_ic_attach", true);

        prepareInputView(context);

        dialog.getWindow().clearFlags(WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE | WindowManager.LayoutParams.FLAG_ALT_FOCUSABLE_IM);
        dialog.getWindow().setSoftInputMode(WindowManager.LayoutParams.SOFT_INPUT_STATE_VISIBLE);

        dialog.setOnDismissListener(this);
        dialog.show();
    }

    protected void prepareInputView(Context context) {
        etInput = dialog.findViewById(ResResolver.getViewId("aihelp_et_input"));
        btnSend = dialog.findViewById(ResResolver.getViewId("aihelp_btn_send"));
        if (etInput != null && btnSend != null) {
            btnSend.setOnClickListener(this);
            btnSend.setEnabled(false);

            etInput.setBackground(Styles.getDrawable(Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.1f), 8));
            Styles.reRenderTextView(etInput, CustomConfig.CustomerService.csInputHint);
            etInput.addTextChangedListener(new TextWatcherWrapper() {
                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    updateSendButtonStatus(context, s);
                }
            });
            etInput.setOnFocusChangeListener(new View.OnFocusChangeListener() {
                @Override
                public void onFocusChange(View v, boolean hasFocus) {
                    if (hasFocus) {
                        SoftInputUtil.showSoftInput(context);
                    } else {
                        SoftInputUtil.hideSoftInput(context, v);
                    }
                }
            });
            etInput.setText(SpUtil.getInstance().getString(SpKeys.INPUT_DRAFT, ""));
            etInput.requestFocus();
        }
    }

    private void updateSendButtonStatus(Context context, CharSequence s) {
        try {
            if (btnSend != null) {
                boolean isSendEnable = !TextUtils.isEmpty(s.toString().trim());
                btnSend.setEnabled(isSendEnable);
                Drawable drawable = Styles.getClickableDrawable(context, "aihelp_svg_ic_send_msg", Color.parseColor(isSendEnable ? CustomConfig.CommonSetting.interactElementTextColor : "#C6C9D7"), isSendEnable);
                btnSend.setImageDrawable(drawable);
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                    if (btnSend.getContext().getResources().getConfiguration().getLayoutDirection() == View.LAYOUT_DIRECTION_RTL) {
                        btnSend.setScaleX(-1); // mirror icon for right-to-left layout
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onClick(View v) {
        if (v.getId() == ResResolver.getViewId("aihelp_btn_send")) {
            String input = etInput.getText().toString().trim();
            if (onUserReplyListener != null) {
                ReplyMessage replyMessage = new ReplyMessage();
                replyMessage.setReplyType(ReplyMessage.REPLY_TYPE_USER);
                replyMessage.setTimeStamp(System.currentTimeMillis());
                replyMessage.setContent(input);
                onUserReplyListener.onUserReply(replyMessage);
            }
            etInput.setText("");
            if (dialog != null) {
                dialog.dismiss();
            }
        }

        if (AppInfoUtil.validateNetwork(v.getContext()) && v.getId() == ResResolver.getViewId("aihelp_iv_attach")) {
            String[] permissionArray = new String[]{Manifest.permission.READ_EXTERNAL_STORAGE};
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
                permissionArray = new String[]{Manifest.permission.READ_MEDIA_IMAGES, Manifest.permission.READ_MEDIA_VIDEO};
            }
            AIHelpPermissions.getInstance()
                    .setFragment(fragment)
                    .setPermissionHandler(this)
                    .setRequestCode(Permission.REQUEST_CODE_TASK_CENTER)
                    .setRequestPermission(permissionArray)
                    .request(fragment.getContext());
        }
    }

    @Permission(requestCode = Permission.REQUEST_CODE_TASK_CENTER)
    public void onPermissionRequested(Permission.Result result, final IPermissionCallback callback) {
        if (fragment != null) {
            switch (result) {
                case GRANTED:
                    AttachmentPicker.getInstance().setPickerHost(fragment)
                            .setAttachmentPickerListener(this).launchPicker(true);
                    break;
                case DENIED:
                    ToastUtil.INSTANCE.showRawSnackBar(fragment.getActivity(), ResResolver.getString("aihelp_permission_denied"), Snackbar.LENGTH_LONG);
                    break;
                case RATIONAL:
                    ToastUtil.INSTANCE.showRawSnackBar(fragment.getActivity(), ResResolver.getString("aihelp_permission_denied"),
                            ResResolver.getString("aihelp_yes"), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    callback.onPermissionRational();
                                }
                            });
                    break;
                case GO_SETTING:
                    ToastUtil.INSTANCE.showRawSnackBar(fragment.getActivity(), ResResolver.getString("aihelp_permission_ignored"),
                            ResResolver.getString("aihelp_permission_settings"), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                                @Override
                                public void onClick(View v) {
                                    callback.onPermissionIgnored();
                                }
                            });
                    break;
                case NONE:
                    AttachmentPicker.getInstance().setPickerHost(fragment)
                            .setAttachmentPickerListener(this).launchPicker(false);
                    break;
            }
        }
    }

    @Override
    public void onPickSuccess(AttachmentPickerFile file, Bundle bundle) {
        String path = file.filePath;
        if (Pattern.compile(RegexDefinition.ANDROID_SUPPORTED_IMAGE).matcher(path).matches()) {
            compressImage(path);
        } else if (Const.TOGGLE_UPLOAD_VIDEO && Pattern.compile(RegexDefinition.ANDROID_SUPPORTED_VIDEO).matcher(path).matches()) {
            uploadFile(ReplyMessage.MSG_TYPE_VIDEO, new File(path));
        } else {
            ToastUtil.INSTANCE.makeRawToast(context, ResResolver.getString("aihelp_resource_not_support"));
        }
    }

    private void compressImage(String path) {
        Luban.with(context).load(path).setCompressListener(new OnCompressListener() {
            @Override
            public void onStart() {
            }

            @Override
            public void onSuccess(File compressedFile) {
                uploadFile(ReplyMessage.MSG_TYPE_IMAGE, compressedFile);
            }

            @Override
            public void onError(Throwable e) {
                uploadFile(ReplyMessage.MSG_TYPE_IMAGE, new File(path));
            }
        }).launch();
    }

    @Override
    public void onPickFailure(int errCode, Long limitSize) {
        if (errCode == AttachmentPicker.ATTACHMENT_FILE_SIZE_LIMIT_EXCEEDED) {
            String msg = ResResolver.getString("aihelp_media_upload_err_size_android");
            try {
                ToastUtil.INSTANCE.makeRawToast(context, String.format(msg, String.valueOf(limitSize / 1048576)));
            } catch (Exception e) {
                e.printStackTrace();
                ToastUtil.INSTANCE.makeRawToast(context, msg + ", < " + (limitSize / 1048576) + "M");
            }
        } else {
            ToastUtil.INSTANCE.makeRawToast(context, "Failed to get specific resource");
        }
    }

    public void uploadFile(int msgType, File file) {
        if (onUserReplyListener != null && file != null) {
            ReplyMessage fileMsg = new ReplyMessage();
            fileMsg.setReplyType(ReplyMessage.REPLY_TYPE_USER);
            fileMsg.setMsgStatus(ReplyMessage.STATUS_SENDING);
            fileMsg.setTimeStamp(System.currentTimeMillis());
            fileMsg.setContent(file.getPath());
            fileMsg.setMsgType(msgType);
            onUserReplyListener.onUserReply(fileMsg);

            UploadFileHelper.INSTANCE.setOnUploadFileListener(new UploadFileHelper.OnUploadFileListener() {
                @Override
                public void onFileUploaded(String fileUrl) {
                    boolean isUrlValid = !TextUtils.isEmpty(fileUrl);
                    fileMsg.setMsgStatus(isUrlValid ? ReplyMessage.STATUS_NORMAL : ReplyMessage.STATUS_FAILURE);
                    fileMsg.setContent(isUrlValid ? fileUrl : fileMsg.getContent());
                    fileMsg.setMsgType(msgType);
                    if (onUserReplyListener != null) {
                        onUserReplyListener.onUserReply(fileMsg);
                    }
                }
            }).performUpload(file);

            if (dialog != null) {
                dialog.dismiss();
            }
        }
    }

    @Override
    public void onDismiss(DialogInterface dialog) {
        AIHelpPermissions.getInstance().recycle();
    }

    public interface OnUserReplyListener {
        void onUserReply(ReplyMessage replyMessage);
    }

}
