package net.aihelp.ui.helper;

import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.ColorStateList;
import android.graphics.Color;
import android.os.Build;
import android.text.TextUtils;
import android.util.SparseArray;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.CompoundButton;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import net.aihelp.common.API;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.UserProfile;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.ui.dialog.AlertDialog;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.bus.Subscribe;
import net.aihelp.core.util.bus.ThreadMode;
import net.aihelp.core.util.bus.event.EventCenter;
import net.aihelp.data.event.OrientationChangeEvent;
import net.aihelp.data.localize.config.FeedbackOpinionHelper;
import net.aihelp.ui.widget.AIHelpButton;
import net.aihelp.ui.widget.AIHelpRatingBar;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.Styles;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

import androidx.appcompat.widget.AppCompatButton;

public class EvaluateNewHelper {

    private AlertDialog rateSupportDialog;

    private LinearLayout llFeedbackLayout;
    private EditText etFeedback;

    private int selectedRateValue = 5;
    private final List<String> selectedOptionIds = new ArrayList<>();
    private String inputFeedback = "";

    public static EvaluateNewHelper getInstance() {
        return new EvaluateNewHelper();
    }

    /**
     * 优先展示解决/未解决弹窗，如果该开关未开启，则展示邀请评分弹窗
     */
    public void show(Context context, final OnConfirmEvaluateListener listener) {
        if (ResponseMqttHelper.isTicketWaitForAskingResolveStatus()) {
            askAboutIssueResolvedStatus(context, listener, null);
        } else {
            showRateSupport(context, listener);
        }
    }

    public void askAboutIssueResolvedStatus(Context context, final OnConfirmEvaluateListener listener,
                                            OnConfirmResolveStatusListener resolveStatusListener) {
        AlertDialog dialog = new AlertDialog.Builder(context)
                .setContentView(ResResolver.getLayoutId("aihelp_dia_issue_solve_status"))
                .setGravity(Gravity.BOTTOM)
                .fromBottom(true)
                .setCancelableOntheOutside(listener != null) // 正常评价客诉时，允许点击外部取消弹窗
                .setWidthAndHeight(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)
                .create();

        RelativeLayout rlLayout = dialog.getView(ResResolver.getViewId("aihelp_rl_issue_solve"));
        int backgroundColor = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                Math.min(CustomConfig.CommonSetting.upperBackgroundAlpha + 0.6, 0.8));
        rlLayout.setBackgroundColor(backgroundColor);

        TextView tvTitle = dialog.getView(ResResolver.getViewId("aihelp_tv_title"));
        Styles.reRenderTextView(tvTitle, ResResolver.getString("aihelp_resolution_request"));

        AIHelpButton tvUnsolved = dialog.getView(ResResolver.getViewId("aihelp_btn_unsolved"));
        tvUnsolved.setText(ResResolver.getString("aihelp_resolution_no"));
        tvUnsolved.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                onSolveStatusConfirmed(context, dialog, listener, resolveStatusListener, false);
            }
        });

        AIHelpButton tvSolved = dialog.getView(ResResolver.getViewId("aihelp_btn_solved"));
        tvSolved.setText(ResResolver.getString("aihelp_resolution_yes"));
        tvSolved.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                onSolveStatusConfirmed(context, dialog, listener, resolveStatusListener, true);
            }
        });

        dialog.show();
    }

    private void onSolveStatusConfirmed(Context context, Dialog dialog, OnConfirmEvaluateListener listener,
                                        OnConfirmResolveStatusListener resolveStatusListener, boolean isResolved) {
        if (resolveStatusListener != null) {
            dialog.dismiss();
            resolveStatusListener.onConfirmResolve(isResolved);
            return;
        }
        if (AppInfoUtil.validateNetwork(context)) {
            try {
                dialog.dismiss();
                if (listener != null) {
                    JSONObject params = new JSONObject();
                    params.put("userId", UserProfile.USER_ID);
                    params.put("deviceId", DeviceUuidFactory.id(context));
                    params.put("resolveType", isResolved ? 1 : 2);
                    AIHelpRequest.getInstance().requestPostByJson(API.ASK_FOR_RESOLVE_STATUS, params, null);
                    ResponseMqttHelper.setTicketWaitForAskingResolveStatus(false);
                    if (ResponseMqttHelper.isTicketWaitForRating()) {
                        showRateSupport(context, listener);
                    } else {
                        onConfirmEvaluate(listener);
                        showThanksAfterEvaluate(context, false);
                    }
                }
            } catch (Exception e) {
                // ignored
            }
        }
    }

    private void showRateSupport(final Context context, final OnConfirmEvaluateListener listener) {

        rateSupportDialog = new AlertDialog.Builder(context)
                .setContentView(ResResolver.getLayoutId("aihelp_dia_rate_support"))
                .setGravity(Gravity.BOTTOM)
                .fromBottom(true)
                .setCancelableOntheOutside(true)
                .setOnDismissListener(new DialogInterface.OnDismissListener() {
                    @Override
                    public void onDismiss(DialogInterface dialog) {
                        EventBus.getDefault().unregister(EvaluateNewHelper.this);
                    }
                })
                .fullWidth()
                .create();

        // 主背景
        int backgroundColor = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                Math.min(CustomConfig.CommonSetting.upperBackgroundAlpha + 0.6, 0.8));
        RelativeLayout rateLayout = rateSupportDialog.getView(ResResolver.getViewId("aihelp_rl_rate_layout"));
        rateLayout.setBackgroundColor(backgroundColor);

        // Title
        TextView tvTitle = rateSupportDialog.getView(ResResolver.getViewId("aihelp_tv_title"));
        Styles.reRenderTextView(tvTitle, CustomConfig.CustomerService.csInviteEvaluate);

        // Rating bar dislike text
        TextView tvRatingDislike = rateSupportDialog.getView(ResResolver.getViewId("aihelp_tv_rating_dislike"));
        Styles.reRenderTextView(tvRatingDislike, ResResolver.getString("aihelp_rate_dissatisfied"));

        // Rating bar like text
        TextView tvRatingLike = rateSupportDialog.getView(ResResolver.getViewId("aihelp_tv_rating_like"));
        Styles.reRenderTextView(tvRatingLike, ResResolver.getString("aihelp_rate_satisfied"));

        llFeedbackLayout = rateSupportDialog.getView(ResResolver.getViewId("aihelp_ll_feedback"));

        SparseArray<JSONArray> opinionArray = FeedbackOpinionHelper.INSTANCE.getOpinionArray();
        resetFeedbackOptions(context, opinionArray.get(selectedRateValue));

        AIHelpRatingBar ratingBar = rateSupportDialog.getView(ResResolver.getViewId("aihelp_rating_bar"));
        if (ratingBar != null) {
            ratingBar.setOnStatusChangedListener(new AIHelpRatingBar.OnStatusChangedListener() {
                @Override
                public void onRateStatusChanged(int newRateValue) {
                    selectedOptionIds.clear();
                    selectedRateValue = newRateValue;
                    resetFeedbackOptions(context, opinionArray.get(newRateValue));
                }
            });
        }

        AIHelpButton btnConfirm = rateSupportDialog.getView(ResResolver.getViewId("aihelp_btn_confirm"));
        btnConfirm.setText(ResResolver.getString("aihelp_yes")).setFullWidth();
        btnConfirm.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppInfoUtil.validateNetwork(v.getContext())) {
                    showThanksAfterEvaluate(context, selectedRateValue == 5);
                    rateSupportDialog.dismiss();
                    onConfirmEvaluate(listener);
                }
            }
        });

        rateSupportDialog.show();

        EventBus.getDefault().register(EvaluateNewHelper.this);

    }

    private void onConfirmEvaluate(OnConfirmEvaluateListener listener) {
        if (listener != null) {
            String input = "";
            if (etFeedback != null) {
                input = etFeedback.getText().toString();
            }
            try {
                JSONArray selectedOptions = new JSONArray();
                JSONArray options = FeedbackOpinionHelper.INSTANCE.getOpinionArray().get(selectedRateValue);
                for (int i = 0; i < selectedOptionIds.size(); i++) {
                    String id = selectedOptionIds.get(i);
                    for (int j = 0; j < options.length(); j++) {
                        JSONObject object = JsonHelper.getJsonObject(options, j);
                        if (id.equals(JsonHelper.optString(object, "id"))) {
                            selectedOptions.put(object);
                        }
                    }
                }
                listener.onConfirmEvaluate(selectedRateValue, input, selectedOptions);
            } catch (Exception e) {
                // ignored
            }
        }
    }

    private void resetFeedbackOptions(Context context, JSONArray options) {
        if (CustomConfig.CustomerService.isEvaluateBadServiceEnable) {
            if (Styles.isLandscape()) {
                prepareFeedbackForLandscape(context, options);
            } else {
                prepareFeedbackForPortrait(context, options);
            }
        }
    }

    private void prepareFeedbackForPortrait(Context context, JSONArray options) {
        llFeedbackLayout.removeAllViews();
        if (options != null && options.length() > 0) {
            for (int i = 0; i < options.length(); i++) {
                llFeedbackLayout.addView(getFeedbackOptionItem(context, JsonHelper.getJsonObject(options, i)));
            }
            prepareFeedbackInput(context);
        }
    }

    private void prepareFeedbackForLandscape(Context context, JSONArray options) {
        llFeedbackLayout.removeAllViews();
        if (options != null && options.length() > 0) {
            for (int i = 0; i < options.length(); i += 2) {
                View previousView = getFeedbackOptionItem(context, JsonHelper.getJsonObject(options, i));
                View afterwardsView = null;
                if (i + 1 < options.length()) {
                    afterwardsView = getFeedbackOptionItem(context, JsonHelper.getJsonObject(options, i + 1));
                }

                LinearLayout linearLayout = new LinearLayout(context);
                linearLayout.setOrientation(LinearLayout.HORIZONTAL);
                linearLayout.addView(previousView);
                if (afterwardsView != null) {
                    linearLayout.addView(afterwardsView);
                }
                llFeedbackLayout.addView(linearLayout);
            }
            prepareFeedbackInput(context);
        }
    }

    private void prepareFeedbackInput(Context context) {
        etFeedback = (EditText) View.inflate(context, ResResolver.getLayoutId("aihelp_layout_feedback_input"), null);
        Styles.reRenderTextView(etFeedback, ResResolver.getString("aihelp_rate_additional_feedback_message"));
        if (!TextUtils.isEmpty(inputFeedback)) {
            etFeedback.setText(inputFeedback);
        }
        etFeedback.setBackground(Styles.getDrawable(Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.1f), 8));
        LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        params.topMargin = (int) Styles.dpToPx(context, 12);
        etFeedback.setLayoutParams(params);
        llFeedbackLayout.addView(etFeedback);
    }

    /**
     * 横屏下：选项分为两列显示 + 最后一个输入框，竖屏下：选项单列显示 + 最后一个输入框
     */
    private View getFeedbackOptionItem(Context context, JSONObject opinion) {
        CheckBox checkBox = (CheckBox) View.inflate(context, ResResolver.getLayoutId("aihelp_layout_feedback_option"), null);

        String id = JsonHelper.optString(opinion, "id");
        String msg = JsonHelper.optString(opinion, "msg");
        Styles.reRenderTextView(checkBox, msg);

        // 在设置监听之前设置勾选（可能是因为屏幕旋转出现的），否则会导致数据重复
        for (int i = 0; i < selectedOptionIds.size(); i++) {
            if (id.equals(selectedOptionIds.get(i))) {
                checkBox.setChecked(true);
                break;
            }
        }

        checkBox.setOnCheckedChangeListener(new CompoundButton.OnCheckedChangeListener() {
            @Override
            public void onCheckedChanged(CompoundButton buttonView, boolean isChecked) {
                if (isChecked) {
                    selectedOptionIds.add(JsonHelper.optString(opinion, "id"));
                } else {
                    selectedOptionIds.remove(JsonHelper.optString(opinion, "id"));
                }
                updateCheckBoxEnableStatus(context);
            }
        });

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            ColorStateList colorStateList = new ColorStateList(
                    new int[][]{new int[]{-android.R.attr.state_checked}, new int[]{android.R.attr.state_checked}},
                    new int[]{Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.1f),
                            Styles.getColor(CustomConfig.CommonSetting.interactElementTextColor)}
            );
            checkBox.setButtonTintList(colorStateList);
        }

        LinearLayout.LayoutParams params = new LinearLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT);
        params.weight = 1;
        checkBox.setLayoutParams(params);

        return checkBox;
    }

    private void showThanksAfterEvaluate(final Context context, boolean isFiveStar) {

        AlertDialog showThanksDialog = new AlertDialog.Builder(context)
                .setContentView(ResResolver.getLayoutId("aihelp_dia_show_thanks"))
                .setGravity(Gravity.BOTTOM)
                .fromBottom(true)
                .setCancelableOntheOutside(true)
                .setWidthAndHeight(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT)
                .create();

        View rlShowThanks = showThanksDialog.getView(ResResolver.getViewId("aihelp_rl_show_thanks"));
        int backgroundColor = Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                Math.min(CustomConfig.CommonSetting.upperBackgroundAlpha + 0.6f, 0.8f));
        rlShowThanks.setBackgroundColor(backgroundColor);

        TextView tvTitle = showThanksDialog.getView(ResResolver.getViewId("aihelp_tv_thanks"));
        Styles.reRenderTextView(tvTitle, ResResolver.getString("aihelp_rate_finished"));

        TextView tvCancel = showThanksDialog.getView(ResResolver.getViewId("aihelp_tv_invite_rate"));
        Styles.reRenderTextView(tvCancel, ResResolver.getString("aihelp_rate_app_hint"),
                Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.5f));

        AIHelpButton tvRate = showThanksDialog.getView(ResResolver.getViewId("aihelp_go_rate"));
        tvRate.setText(ResResolver.getString("aihelp_rate_button"));
        tvRate.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                AppInfoUtil.goRateApp(context);
                showThanksDialog.dismiss();
            }
        });

        View goRateLayout = showThanksDialog.getView(ResResolver.getViewId("aihelp_ll_go_rate"));
        goRateLayout.setVisibility(ResponseMqttHelper.isAppRatable() && isFiveStar ? View.VISIBLE : View.GONE);

        showThanksDialog.show();
    }

    private void updateCheckBoxEnableStatus(Context context) {
        int childCount = llFeedbackLayout.getChildCount();
        if (Styles.isLandscape()) {
            for (int i = 0; i < childCount; i++) {
                View childView = llFeedbackLayout.getChildAt(i);
                if (childView instanceof LinearLayout) {
                    LinearLayout layout = (LinearLayout) childView;
                    for (int j = 0; j < layout.getChildCount(); j++) {
                        View view = layout.getChildAt(j);
                        if (view instanceof CheckBox) {
                            CheckBox checkBox = (CheckBox) view;
                            if (!checkBox.isChecked()) {
                                checkBox.setEnabled(selectedOptionIds.size() < CustomConfig.CustomerService.csEvaluateMaxCount);
                            }
                        }
                    }
                }
            }
        } else {
            for (int i = 0; i < childCount; i++) {
                View childView = llFeedbackLayout.getChildAt(i);
                if (childView instanceof CheckBox) {
                    CheckBox checkBox = (CheckBox) childView;
                    if (!checkBox.isChecked()) {
                        checkBox.setEnabled(selectedOptionIds.size() < CustomConfig.CustomerService.csEvaluateMaxCount);
                    }
                }
            }
        }

    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEventComing(EventCenter event) {
        if (event instanceof OrientationChangeEvent && etFeedback != null) {
            inputFeedback = etFeedback.getText().toString();
            SparseArray<JSONArray> opinionArray = FeedbackOpinionHelper.INSTANCE.getOpinionArray();
            resetFeedbackOptions(etFeedback.getContext(), opinionArray.get(selectedRateValue));
        }
    }

    public interface OnConfirmEvaluateListener {
        boolean onConfirmEvaluate(int rate, String input, JSONArray selected);
    }

    public interface OnConfirmResolveStatusListener {
        void onConfirmResolve(boolean isResolved);
    }

}
