package net.aihelp.ui.helper;

import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;

import net.aihelp.common.Const;
import net.aihelp.config.AIHelpContext;
import net.aihelp.utils.FileUtil;

import java.lang.ref.WeakReference;

import androidx.fragment.app.Fragment;


public class AttachmentPicker {

    public static final int ATTACHMENT_TYPE_MEDIA = 1;
    public static final int ATTACHMENT_TYPE_FILE = 2;
    public static final int ATTACHMENT_TYPE_RPA = 3;

    public static final int ATTACHMENT_FILE_NOT_FOUND = -1;
    public static final int UNSUPPORTED_ATTACHMENT_TYPE = -2;
    public static final int ATTACHMENT_FILE_SIZE_LIMIT_EXCEEDED = -3;
    public static final int NO_APPS_TO_OPEN_ATTACHMENTS_INTENT = -4;
    public static final int INVALID_URI = -5;

    private static final long MAX_ATTACHMENT_FILE_SIZE_LIMIT = 26214400L;

    private Bundle extraData;
    private int pickType;

    private WeakReference<Fragment> pickerHostRef;
    private WeakReference<AttachmentPickerListener> attachmentPickerListenerRef;

    public <T extends Fragment> AttachmentPicker setPickerHost(T fragment) {
        pickerHostRef = new WeakReference<>(fragment);
        return this;
    }

    public <T extends AttachmentPickerListener> AttachmentPicker setAttachmentPickerListener(T listener) {
        if (listener != null) {
            attachmentPickerListenerRef = new WeakReference<>(listener);
        }
        return this;
    }

    public void launchPicker(boolean hasPermission) {
        try {
            if (pickerHostRef != null) {
                Fragment fragment = pickerHostRef.get();
                if (fragment != null && fragment.getActivity() != null) {
                    fragment.startActivityForResult(AttachmentDataProvider.getIntentForMedia(hasPermission),
                            hasPermission ? AttachmentDataProvider.PICK_ATTACHMENT_REQUEST_ID :
                                    AttachmentDataProvider.PICK_ATTACHMENT_WITHOUT_PERMISSIONS_REQUEST_ID);
                }
            }
        } catch (ActivityNotFoundException ignored) {
            this.sendPickFailure(NO_APPS_TO_OPEN_ATTACHMENTS_INTENT, null);
        }
    }

    public void onAttachmentPickRequestResult(int requestCode, Intent intent) {
        Uri uri = intent.getData();
        switch (requestCode) {
            case AttachmentDataProvider.PICK_ATTACHMENT_REQUEST_ID:
                processUriForAttachment(uri);
                break;
            case AttachmentDataProvider.PICK_ATTACHMENT_WITHOUT_PERMISSIONS_REQUEST_ID:
                Context context = AIHelpContext.getInstance().getContext();
                context.grantUriPermission(context.getPackageName(), uri, Intent.FLAG_GRANT_READ_URI_PERMISSION);
                processUriForAttachment(uri);
                break;
        }
    }

    private void processUriForAttachment(Uri uri) {
        if (uri != null) {
            Context context = AIHelpContext.getInstance().getContext();
            if (FileUtil.doesFileFromUriExistAndCanRead(uri, context)) {
                AttachmentPickerFile file = AttachmentHelper.createPickFileFromUri(uri);
                Long fileSize = file.originalFileSize;
                long limitSize = Math.max(Const.LIMIT_UPLOADING_VIDEO, MAX_ATTACHMENT_FILE_SIZE_LIMIT);
                if (file.attachmentType == AttachmentHelper.PICK_VIDEO && fileSize != null && fileSize > limitSize
                        || file.attachmentType == AttachmentHelper.PICK_IMAGE && fileSize != null && fileSize > limitSize) {
                    sendPickFailure(ATTACHMENT_FILE_SIZE_LIMIT_EXCEEDED, limitSize);
                } else {
                    sendPickSuccess(file, extraData);
                }

            } else {
                sendPickFailure(ATTACHMENT_FILE_NOT_FOUND, null);
            }
        } else {
            sendPickFailure(INVALID_URI, null);
        }
    }

    private void sendPickSuccess(AttachmentPickerFile attachmentPickerFile, Bundle extraData) {
        AttachmentPickerListener attachmentPickerListener = (AttachmentPickerListener) this.attachmentPickerListenerRef.get();
        if (attachmentPickerListener != null) {
            attachmentPickerListener.onPickSuccess(attachmentPickerFile, extraData);
        }

    }

    private void sendPickFailure(int errorCode, Long data) {
        AttachmentPickerListener attachmentPickerListener = (AttachmentPickerListener) this.attachmentPickerListenerRef.get();
        if (attachmentPickerListener != null) {
            attachmentPickerListener.onPickFailure(errorCode, data);
        }
    }

    public interface AttachmentPickerListener {

        void onPickSuccess(AttachmentPickerFile var1, Bundle var2);

        void onPickFailure(int errCode, Long limitSize);
    }

    private static final class LazyHolder {
        static final AttachmentPicker INSTANCE = new AttachmentPicker();

        private LazyHolder() {
        }

    }

    public static AttachmentPicker getInstance() {
        return AttachmentPicker.LazyHolder.INSTANCE;
    }

}

