package net.aihelp.ui.cs;

import android.content.res.Configuration;
import android.graphics.Color;
import android.graphics.drawable.Drawable;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;


import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.core.mvp.IPresenter;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.ui.BaseFragment;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.bus.Subscribe;
import net.aihelp.core.util.bus.ThreadMode;
import net.aihelp.data.event.SearchViewVisibilityChangeEvent;
import net.aihelp.data.event.PrepareMessageTimeStampEvent;
import net.aihelp.data.event.SupportActionEvent;
import net.aihelp.data.event.UpdateTitleEvent;
import net.aihelp.data.logic.MqttCallbackImpl;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.wrapper.TextWatcherWrapper;
import net.aihelp.ui.widget.AIHelpChatRecyclerView;
import net.aihelp.utils.KeyboardChangeHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.Styles;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.AppCompatImageButton;
import androidx.core.graphics.drawable.DrawableCompat;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.vectordrawable.graphics.drawable.VectorDrawableCompat;


public abstract class BaseCSFragment<P extends IPresenter> extends BaseFragment<P> implements View.OnClickListener {

    AIHelpChatRecyclerView rvMsgList;
    MessageListAdapter mAdapter;

    ViewGroup bottomLayout;
    EditText etInput;
    AppCompatImageButton btnSend;
    IMqttCallback mqttCallback;

    boolean isAlwaysShowSupportInElva;
    private KeyboardChangeHelper helper;
    boolean isOperateBot;
    boolean isOperateHuman;

    private ConversationMsg mTimeStampMsg;

    private String titleIcon;
    private String titleText;

    @Override
    protected void getBundleBeforeDataPrepared(Bundle extras) {
        isOperateBot = extras.getBoolean(IntentValues.BOT_EMBEDDED_IN_OPERATION, false);
        isOperateHuman = extras.getBoolean(IntentValues.HUMAN_EMBEDDED_IN_OPERATION, false);
        titleIcon = CustomConfig.CustomerService.csNavigationBarTitleIcon;
        titleText = CustomConfig.CustomerService.csNavigationTitle;
    }

    @Override
    public void onResume() {
        super.onResume();
        EventBus.getDefault().post(new UpdateTitleEvent(UpdateTitleEvent.CUSTOM_SERVICE, titleIcon, titleText));
        EventBus.getDefault().post(new SearchViewVisibilityChangeEvent(false));
    }

    @Override
    protected void initEventAndData(View contentView) {
        EventBus.getDefault().post(new SupportActionEvent(IntentValues.HIDE_SUPPORT_ACTION));

        mqttCallback = MqttCallbackImpl.getInstance();
        bottomLayout = get("aihelp_chat_bottom_layout");
        etInput = get("aihelp_et_input");
        btnSend = get("aihelp_btn_send");
        rvMsgList = get("aihelp_rv_msg_list");

        rvMsgList.setBackgroundColor(Color.TRANSPARENT);
        mAdapter = new MessageListAdapter(getContext());
        rvMsgList.setAdapter(mAdapter);

        final LinearLayoutManager linearLayoutManager = new LinearLayoutManager(getContext());
        rvMsgList.setLayoutManager(linearLayoutManager);

//        if (this.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
//            SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BACKGROUND_PORTRAIT, rvMsgList);
//        } else {
//            SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BACKGROUND_LANDSCAPE, rvMsgList);
//        }
//        SkinHelper.updateBackground(SkinHelper.SKIN_INPUT_AREA, etInput);
//        SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BOTTOM_AREA, bottomLayout);
//        SkinHelper.updateIcon(SkinHelper.SKIN_SEND_MSG, btnSend);

        bottomLayout.setBackgroundColor(Styles.getColorWithAlpha(CustomConfig.CommonSetting.upperBackgroundColor,
                CustomConfig.CommonSetting.upperBackgroundAlpha));

        etInput.setBackground(Styles.getDrawable(Styles.getColorWithAlpha(CustomConfig.CommonSetting.textColor, 0.1f), 8));
        Styles.reRenderTextView(etInput, CustomConfig.CustomerService.csInputHint);
        etInput.addTextChangedListener(new TextWatcherWrapper() {
            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                onTextChange(s);
                updateSendButtonStatus(s);
            }
        });

        helper = new KeyboardChangeHelper(getView());
        helper.addListener();
        helper.addOnKeyboardShowListener(new KeyboardChangeHelper.OnKeyboardShowListener() {
            @Override
            public void onKeyboardShow() {
                if (!linearLayoutManager.getStackFromEnd()) {
                    int firstVisibleItemPosition = linearLayoutManager.findFirstCompletelyVisibleItemPosition();
                    if (firstVisibleItemPosition > 0) linearLayoutManager.setStackFromEnd(true);
                }
                scrollRecyclerViewToEnd();
            }

            @Override
            public void onKeyboardHide() {
            }
        });

        btnSend.setOnClickListener(this);
        btnSend.setEnabled(false);

    }

    private void updateSendButtonStatus(CharSequence s) {
        try {
            if (btnSend != null) {
                boolean isSendEnable = !TextUtils.isEmpty(s.toString().trim());
                btnSend.setEnabled(isSendEnable);
                Drawable drawable = Styles.getClickableDrawable(getContext(), "aihelp_svg_ic_send_msg",
                        Color.parseColor(isSendEnable ? CustomConfig.CommonSetting.interactElementTextColor : "#C6C9D7"),
                        isSendEnable);
                btnSend.setImageDrawable(drawable);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    protected void onTextChange(CharSequence s) {

    }

    public void updateChatList(ConversationMsg msg) {

        if (CustomConfig.CustomerService.isMessageTimestampVisible && msg.getMsgType() != ConversationMsg.TYPE_TIMESTAMP) {
            prepareTimeStamp(msg.getTimeStamp());
        }

        if (msg.getMsgType() == ConversationMsg.TYPE_USER_TEXT ||
                msg.getMsgType() == ConversationMsg.TYPE_USER_TEXT_BOT) {
            etInput.setText("");
            updateSendButtonStatus("");
        }
        mAdapter.update(msg);
        scrollRecyclerViewToEnd();
    }

    private void scrollRecyclerViewToEnd() {
        rvMsgList.post(new Runnable() {
            @Override
            public void run() {
                rvMsgList.scrollToPosition(mAdapter.getItemCount() - 1);
            }
        });
    }

    public void updateChatListClearly(ConversationMsg msg) {
        ArrayList<ConversationMsg> conversationList = new ArrayList<>();
        conversationList.add(msg);
        updateChatList(conversationList);
    }

    public void updateChatList(List<ConversationMsg> msg) {
        mAdapter.update(msg, true);
        scrollRecyclerViewToEnd();
    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        if (rvMsgList != null) {
            SoftInputUtil.hideSoftInput(getContext(), rvMsgList);
            rvMsgList.setBackgroundColor(Color.TRANSPARENT);
            if (mAdapter != null) mAdapter.notifyDataSetChanged();
        }
    }

    public void onMqttLogin(List<ConversationMsg> mqttReplyMsg) {
        btnSend.setEnabled(true);
        // Must ensure current page is visible, or the page status will be messed up.
        if (isVisible()) {
            if (this instanceof ElvaBotFragment && getArguments() != null) {
                isAlwaysShowSupportInElva = getArguments().getBoolean(IntentValues.CONTACT_US_ALWAYS_ONLINE);
                if (isAlwaysShowSupportInElva) {
                    EventBus.getDefault().post(new SupportActionEvent(IntentValues.SHOW_SUPPORT_ACTION,
                            IntentValues.SUPPORT_ACTION_MSG));
                }
                if (!TextUtils.isEmpty(Const.CUSTOM_STORY_NODE) || !Const.isLocalWelcomeAvailable) {
                    updateChatList(mqttReplyMsg);
                }
            }

            if (this instanceof ConversationFragment) {
                updateChatList(mqttReplyMsg);
            }
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (helper != null) {
            helper.removeListener();
        }
    }

    public void updateNetCheckingStatus(boolean startChecking) {

    }

    public boolean isNetCheckingInProgress() {
        return false;
    }

    @Override
    protected boolean isBindEventBus() {
        return true;
    }

    private void prepareTimeStamp(long timeStamp) {
        if (mTimeStampMsg == null || timeStamp - Long.parseLong(mTimeStampMsg.getMsgContent()) > 5 * 60 * 1000) {
            mTimeStampMsg = new ConversationMsg(ConversationMsg.TYPE_TIMESTAMP, ConversationMsg.STATUS_SUCCESS);
            mTimeStampMsg.setMsgContent(timeStamp + "");
            updateChatList(mTimeStampMsg);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEventComing(PrepareMessageTimeStampEvent event) {
        if (CustomConfig.CustomerService.isMessageTimestampVisible && isVisible()) {
            ConversationMsg newMsg = (ConversationMsg) event.getEvent();
            prepareTimeStamp(newMsg.getTimeStamp());
        }
    }

}
