package net.aihelp.data.logic;

import android.content.Context;
import android.os.Handler;
import android.os.Message;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.common.API;
import net.aihelp.common.ConfigValues;
import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.ConversationConfig;
import net.aihelp.config.UserConfig;
import net.aihelp.core.mvp.AbsPresenter;
import net.aihelp.core.mvp.IView;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.data.local.InitRepository;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.data.localize.data.OperateHelper;
import net.aihelp.data.model.init.InitEntity;

import net.aihelp.ui.helper.MessageSyncHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.listener.OnAIHelpInitializedCallback;
import net.aihelp.ui.listener.OnAIHelpSessionCloseCallback;
import net.aihelp.ui.listener.OnAIHelpSessionOpenCallback;
import net.aihelp.ui.listener.OnSpecificUrlClickedCallback;
import net.aihelp.ui.listener.OnMessageCountArrivedCallback;
import net.aihelp.ui.listener.OnNetworkCheckResultCallback;
import net.aihelp.ui.listener.OnOperationUnreadChangedCallback;
import net.aihelp.ui.listener.OnSpecificFormSubmittedCallback;
import net.aihelp.utils.DateFormatUtil;
import net.aihelp.utils.DeviceUuidFactory;
import net.aihelp.utils.LocaleUtil;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.TLog;

import org.json.JSONException;
import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import androidx.annotation.NonNull;


public class InitPresenter extends AbsPresenter<IView, InitRepository> {

    private OnAIHelpInitializedCallback mInitListener;
    private final RetryInitHandler mRetryInitHandler;

    public InitPresenter(Context context, String appKey, String domain, String appId, String language) {
        super(context);
        mRetryInitHandler = new RetryInitHandler(this);
        mRepo.saveInitConfig(appKey, domain, appId, language);
    }

    public void updateUserProfile(final UserConfig config) {
        if (config == null) return;
        mRepo.saveUserProfileConfig(config);
        Const.TOGGLE_FETCH_MESSAGE = true;
        ResponseMqttHelper.resetTicketStatusFlags();
        if (config.isSyncCrmInfo()) {
            final String latelyCrmInfo = String.format("%s|%s", config.getUserId(), config.getUserTags());
            String cachedCrmInfo = mSp.getString(SpKeys.CRM_USER_INFO + config.getUserId());
            if (!latelyCrmInfo.equals(cachedCrmInfo)) {
                try {
                    JSONObject json = new JSONObject();
                    json.put("userName", config.getUserName());
                    json.put("uId", config.getUserId());
                    json.put("tags", config.getUserTags());
                    post(API.SYNC_VIP_INFO, json, new ReqCallback<String>() {
                        @Override
                        public void onReqSuccess(String result) {
                            mRepo.saveCrmInfo(config.getUserId(), config.getUserTags());
                        }
                    });
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }
        if (config.getPushToken() != null) {
            postCrmPushTokenInfo(config.getPushToken(), config.getPushPlatform().getValue());
        }
    }

    public void updateConversationFields(ConversationConfig config) {
        mRepo.updateConversationFields(config.getWelcomeMessage(), config.getStoryNode());
    }

    public void doInitForAIHelp() {
        // AIHelpSupport APIs are all running on UI thread, so if there are cached init responses,
        // the initialization will be succeed directly, but because of the main looper message
        // queue, the init callback haven't been set.
        // Transfer init task to another message queue and delayed for 0.5s to make sure init callback can work well.
        ApiExecutorFactory.getHandlerExecutor().runAsync(new Runnable() {
            @Override
            public void run() {
                goLogDauStatus();
                requestInit(true);
                MessageSyncHelper.syncLogMessage();
            }
        });
    }

    private void goLogDauStatus() {
        long initTime = mSp.getLong(SpKeys.LOG_DAU_TIME);
        if (!DateFormatUtil.isToday(initTime)) {
            try {
                JSONObject dauParams = new JSONObject();
                // DO NOT try to change this deviceId EVER, or the DAU data will be messed, which will
                // results in pricing-related issues.
                dauParams.put("deviceid", DeviceUuidFactory.id(mContext));
                post(API.LOG_DAU_URL, dauParams, new ReqCallback<String>() {
                    @Override
                    public void onReqSuccess(String result) {
                        mRepo.saveDAULogTime(System.currentTimeMillis());
                    }
                });
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private void requestInit(boolean requireCallback) {
        String spPrefix = String.format("%s_%s_", Const.APP_ID, Const.ORIGINAL_LANGUAGE);
        final long lastInitTime = mSp.getLong(spPrefix + SpKeys.LAST_INIT_TIME, System.currentTimeMillis());
        int requestLimit = mSp.getInt(spPrefix + SpKeys.INIT_REQUEST_LIMIT);
        final String cachedInitData = mSp.getString(spPrefix + SpKeys.CACHE_INIT_RESPONSE);

        // reset localize data to ensure data's accuracy
        LocalizeHelper.resetLocalizeData();

        if (TextUtils.isEmpty(cachedInitData) || requestLimit < 0 || System.currentTimeMillis() - lastInitTime > requestLimit) {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("domain", API.HOST_URL);
                AIHelpRequest.getInstance().requestGetByAsync(API.INIT_URL, jsonObject, new ReqCallback<String>() {

                    @Override
                    public void onAsyncReqSuccess(String result) {
                        mRetryInitHandler.removeCallbacksAndMessages(null);
                        cacheInitResponse(requireCallback, result, System.currentTimeMillis());
                    }

                    @Override
                    public void onAsyncFailure(String url, int errorCode, String errorMsg) {
                        if (TextUtils.isEmpty(cachedInitData)) {
                            mRetryInitHandler.retryInitRequest(errorCode);
                        } else {
                            cacheInitResponse(requireCallback, cachedInitData, lastInitTime);
                        }
                    }

                });
            } catch (Exception e) {
                e.printStackTrace();
            }
        } else {
            cacheInitResponse(requireCallback, cachedInitData, lastInitTime);
        }
    }

    private void cacheInitResponse(boolean requireCallback, String initResponse, long lastInitTime) {
        try {
            InitEntity initEntity = JsonHelper.toJavaObject(initResponse, InitEntity.class);
            if (initEntity != null) {
                mRepo.prepareInitData(initEntity, initResponse, lastInitTime);
                ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        AIHelpContext.successfullyInit.set(true);
                        if (requireCallback && mInitListener != null) {
                            mInitListener.onAIHelpInitialized();
                            mInitListener = null;
                        }
                    }
                });
            } else {
                AIHelpLogger.error("init error", new UnknownError(initResponse));
                Log.e("AIHelp", String.format("AIHelp SDK init failed, %s", initResponse));
            }
        } catch (Exception e) {
            // ignored
        }
    }

    public void postCrmPushTokenInfo(final String pushToken, final int pushPlatform) {
        if (!TextUtils.isEmpty(pushToken) && pushPlatform > 0) {
            mRepo.saveMqttPushInfo(pushToken, pushPlatform);
            if (Const.TOGGLE_CRM_TOKEN) {
                final String latelyCrmPushInfo = String.format("%s|%s|%s", UserProfile.USER_ID, pushToken, pushPlatform);
                String cachedCrmPushInfo = mSp.getString(SpKeys.PUSH_INFO_CRM);
                if (!latelyCrmPushInfo.equals(cachedCrmPushInfo)) {
                    try {
                        JSONObject params = new JSONObject();
                        params.put("token", pushToken);
                        params.put("pushTypeId", pushPlatform);
                        params.put("playerId", UserProfile.USER_ID);
                        params.put("playerName", UserProfile.USER_NAME);
                        params.put("language", Const.CORRECT_LANGUAGE);
                        params.put("deviceId", DeviceUuidFactory.id(mContext));
                        post(API.CRM_TOKEN_URL, params, new ReqCallback<String>() {
                            @Override
                            public void onReqSuccess(String result) {
                                mSp.put(SpKeys.PUSH_INFO_CRM, latelyCrmPushInfo);
                            }
                        });
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    public void updateSDKLanguage(final String sdkLanguage) {
        String formatLanguage = LocaleUtil.getFormatLanguage(sdkLanguage);
        if (!Const.ORIGINAL_LANGUAGE.equals(formatLanguage)) {
            // reset init status to avoid opening AIHelp UI before init success
            AIHelpContext.successfullyInit.set(false);

            // update memory language
            Const.ORIGINAL_LANGUAGE = formatLanguage;
            Const.CORRECT_LANGUAGE = Const.ORIGINAL_LANGUAGE;

            // request init to fetch latest data after updating language
            requestInit(false);
        } else {
            Log.e("TAG", "AIHelp is using " + formatLanguage + " as global language already!");
        }
    }

    public void setUploadLogPath(String logPath) {
        if (TextUtils.isEmpty(logPath)) return;
        if (logPath.endsWith(".txt") || logPath.endsWith(".log") || logPath.endsWith(".bytes")) {
            mRepo.setUploadLogPath(logPath);
        } else {
            TLog.e("Unsupported type, expected .txt, .log or .bytes file");
        }
    }

    public void setNetworkCheckHostAddress(String hostAddress, OnNetworkCheckResultCallback listener) {
        if (TextUtils.isEmpty(hostAddress)) {
            hostAddress = "aihelp.net";
        } else {
            hostAddress = hostAddress.replace("http://", "")
                    .replace("https://", "");
        }
        mRepo.setNetworkCheckHostAddress(hostAddress);
        Const.sCheckResultListener = listener;
    }

    private void fetchUnreadMessageCount() {
        try {
            if (Const.IS_SDK_SHOWING) {
                TLog.e("AIHelp", "AIHelp session is visible to user, do not need fetch for unread messages.");
                return;
            }

            if (!Const.TOGGLE_FETCH_MESSAGE) {
                TLog.e("AIHelp", String.format("Current user(%s) does not have any active tickets at present.", UserProfile.USER_ID));
                return;
            }

            JSONObject params = new JSONObject();
            params.put("appid", Const.APP_ID);
            params.put("uid", UserProfile.USER_ID);
            if (UserProfile.USER_ID.equals(DeviceUuidFactory.id(mContext))) {
                TLog.e("AIHelp", "The userId you're using for unread message polling is AIHelp's generated deviceId, " +
                        "please verify if this is what you want.");
            }
            AIHelpRequest.getInstance().requestGetByAsync(API.FETCH_NEW_MSG, params, new ReqCallback<String>() {
                @Override
                public void onReqSuccess(String result) {
                    try {
                        if (!TextUtils.isEmpty(result)) {
                            JSONObject jsonObject = new JSONObject(result);
                            Const.TOGGLE_FETCH_MESSAGE = jsonObject.optBoolean("isHaveChat");
                            int newCount = jsonObject.optInt("cs_message_count");
                            int cachedCount = Math.max(0, mSp.getInt(UserProfile.USER_ID));
                            if (Const.sMessageListener != null) {
                                Const.sMessageListener.onMessageCountArrived(Math.max(0, newCount - cachedCount));
                            }
                        }
                    } catch (Exception e) {
                        TLog.e("Unread msg count polling failed, because " + e.toString());
                    }
                }
            });
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    private ScheduledFuture<?> unreadMsgFuture;

    public void startUnreadMessagePolling(OnMessageCountArrivedCallback listener) {
        if (SpUtil.getInstance().getBoolean(SpKeys.TOGGLE_LOG)) Const.LIMIT_CHECKING_UNREAD = 10;
        if (listener != null && unreadMsgFuture == null) {
            if (Const.TOGGLE_OPEN_UNREAD_MSG && Const.LIMIT_CHECKING_UNREAD > 0 && !TextUtils.isEmpty(UserProfile.USER_ID)) {
                Const.sMessageListener = listener;
                unreadMsgFuture = Executors.newScheduledThreadPool(1).scheduleAtFixedRate(new Runnable() {
                    @Override
                    public void run() {
                        fetchUnreadMessageCount();
                    }
                }, 0, Const.LIMIT_CHECKING_UNREAD, TimeUnit.SECONDS);
            }
        }
    }

    public void loadUpListeners(OnAIHelpInitializedCallback listener) {
        if (listener != null && mInitListener == null) {
            mInitListener = listener;
        }
    }

    public void setOnSpecificFormSubmittedCallback(OnSpecificFormSubmittedCallback callback) {
        Const.sSpecificFormSubmittedListener = callback;
    }

    public void setOnAIHelpSessionOpenCallback(OnAIHelpSessionOpenCallback callback) {
        Const.sSessionOpenListener = callback;
    }

    public void setOnAIHelpSessionCloseCallback(OnAIHelpSessionCloseCallback callback) {
        Const.sSessionCloseListener = callback;
    }

    public void setOnSpecificUrlClickedCallback(OnSpecificUrlClickedCallback callback) {
        Const.sOnSpecificUrlClickedListener = callback;
    }

    public void setOnOperationUnreadChangedCallback(OnOperationUnreadChangedCallback callback) {
        Const.sOperationUnreadListener = callback;
        ApiExecutorFactory.getHandlerExecutor().runAsyncDelayed(new Runnable() {
            @Override
            public void run() {
                if (Const.sOperationUnreadListener != null) {
                    Const.sOperationUnreadListener.onOperationUnreadChanged(OperateHelper.INSTANCE.haveUnreadArticles());
                }
            }
        }, 1000);
    }

    public static class RetryInitHandler extends Handler {

        private int retryCount;
        private final InitPresenter initPresenter;

        public RetryInitHandler(InitPresenter initPresenter) {
            super();
            this.initPresenter = initPresenter;
        }

        public void retryInitRequest(int code) {
            if ((code >= 200 && code < 300) || (code >= 400 && code < 600)) return;
            if (retryCount++ < 3) {
                sendEmptyMessageDelayed(0, 20 * 1000);
            } else {
                removeCallbacksAndMessages(null);
            }
        }

        @Override
        public void handleMessage(@NonNull Message msg) {
            initPresenter.requestInit(true);
        }
    }


}
