package net.aihelp.data.localize.data;

import android.text.TextUtils;

import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.data.model.op.OperateArticle;
import net.aihelp.data.model.op.OperateSection;
import net.aihelp.utils.FileUtil;
import net.aihelp.utils.SpUtil;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public enum OperateHelper {

    INSTANCE;

    // 用来处理未读文章的数据
    private String historicArticles = "";
    private JSONArray latelyFlatArticles = new JSONArray();

    // 未加工的原始数据
    private Map<String, JSONArray> rawArticleMap = new HashMap<>();

    // 加工处理过的数据源
    private List<OperateSection> rootSections = new ArrayList<>();
    private Map<String, List<OperateArticle>> opArticlesMap = new HashMap<>();

    // ******************************** 对外暴露的公共方法 ********************************
    public List<OperateSection> getOperateSections() {
        return rootSections;
    }

    public List<OperateArticle> getOperateArticles(String sectionId) {
        return opArticlesMap.get(sectionId);
    }

    public synchronized void prepareHistoricDataSource() {
        if (Const.sOperationUnreadListener != null) {
            try {
                // 在下载新的运营模块数据之前，先取出本地已经存在的旧数据
                // 如果最新的文件名和缓存的文件名一致，说明没有重新发布过，也就不需要进行比对；程序会在后面尝试对比时回调给调用者
                String lastCachedOpFile = SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.LAST_CACHED_OPERATE_FILE));
                if (!lastCachedOpFile.equals(Const.OP_FILE)) {
                    // 需要将文件名称缓存到本地，此时全局的 OP_FILE 已经是最新的了，根据缓存的文件名去找对应的文件
                    String filePath = LocalizeUtil.getFileLocation(LocalizeHelper.FLAG_OP, lastCachedOpFile);
                    String cacheContent = FileUtil.getContentFromFile(filePath);
                    if (!TextUtils.isEmpty(cacheContent)) {
                        JSONArray cacheSectionArray = JsonHelper.getJsonArray(new JSONObject(cacheContent), "faqlist");
                        // 将所有运营文章都放到同一个数组中，用来与最新的数据进行比对
                        StringBuilder builder = new StringBuilder();
                        for (int i = 0; i < cacheSectionArray.length(); i++) {
                            JSONObject section = JsonHelper.getJsonObject(cacheSectionArray, i);
                            JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
                            if (faqs.length() > 0) {
                                for (int j = 0; j < faqs.length(); j++) {
                                    JSONObject article = JsonHelper.getJsonObject(faqs, j);
                                    builder.append(article.optString("kmMainid"))
                                            .append(",").append(article.optString("lastUpdateTime")).append("|");
                                }
                            }
                        }
                        historicArticles = builder.toString();
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * 切换语言时重置数据
     * 注意需要把全局保存的前端化地址也清空，否则就会导致文件夹错乱
     * 这样请求的时候 url 地址就不是 .json 结尾的数据，会直接去请求接口获取数据，数据不会有问题
     *
     * FAQ 数据与运营模块数据不同，FAQ 有可能直接打开页面下载，所以在切换语言时必须做清空处理，否则数据就有可能乱掉
     * 比如，从中文切换到英文，但是打开 FAQ 后，显示的还是中文，此时下载动作在后台执行，需要退出再进入才能变成正确的英文
     *
     * 所以如下的清空动作对于 FAQ 是必须的，但对于 OP 来说是可选的
     *
     */
    public void reset() {
        rootSections.clear();
        opArticlesMap.clear();
        Const.OP_FILE = "";
    }

    public synchronized void prepareDataSource() {
        try {
            // 从本地读取 FAQ 文件
            String content = FileUtil.getContentFromFile(LocalizeUtil.getFileLocation(LocalizeHelper.FLAG_OP));
            if (!TextUtils.isEmpty(content)) {
                JSONArray rawSectionArray = JsonHelper.getJsonArray(new JSONObject(content), "faqlist");

                // 运营模块分类列表
                prepareRootSectionList(rawSectionArray);

                // 运营文章问题列表字典：<文章所属分类ID, List<OperateEntity>>
                prepareSectionToArticlesMap(rawSectionArray);

                // 将所有运营文章都放到同一个集合中
                flatRawArticleJsonArray(rawSectionArray);

                // 尝试与 historicFlatArticles 做对比，找到更新或新增的文章，标记未读
                tryToFindUpdatedArticles();

                // 将最新的运营文章 json 文件名缓存起来，用于下次比对的时候获取文件
                SpUtil.getInstance().put(getLanguageBasedKey(SpKeys.LAST_CACHED_OPERATE_FILE), Const.OP_FILE);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 找到未读的运营文章：
     * 1、可能是新添加的文章，表现为 id 是新的；
     * 2、可能是旧文章做了改动，表现为 id 相同但是最后更新时间不同；
     */
    private void tryToFindUpdatedArticles() {
        if (Const.sOperationUnreadListener != null) {
            if (!TextUtils.isEmpty(historicArticles)) {
                StringBuilder unreadArticlesBuilder = new StringBuilder();
                for (int i = 0; i < latelyFlatArticles.length(); i++) {
                    JSONObject latestArticle = JsonHelper.getJsonObject(latelyFlatArticles, i);
                    String id = latestArticle.optString("kmMainid");
                    String updateTime = latestArticle.optString("lastUpdateTime");
                    if (historicArticles.contains(id + "," + updateTime)) {
                        continue;
                    }
                    // 如果可以执行到这里，说明无法在字符串中匹配到，就应该记为未读
                    unreadArticlesBuilder.append(id).append(",");
                }
                // 循环结束后，将所有未读文章都缓存起来
                SpUtil.getInstance().put(getLanguageBasedKey(SpKeys.UNREAD_ARTICLES), unreadArticlesBuilder.toString());
            }
            // 无论是否有比较逻辑，都检查是否有未读文章，然后将结果回调给调用者
            String unreadArticles = SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.UNREAD_ARTICLES));
            Const.sOperationUnreadListener.onOperationUnreadChanged(!TextUtils.isEmpty(unreadArticles.toString()));
        }
    }

    public OperateArticle getArticleById(String faqId) {
        return getArticleById(null, faqId);
    }

    public OperateArticle getArticleById(String sectionId, String faqId) {
        if (TextUtils.isEmpty(sectionId)) {
            return filterFaqById(latelyFlatArticles, faqId);
        }
        return filterFaqById(rawArticleMap.get(sectionId), faqId);
    }

    public void updateOperateFaqUnreadStatus(String articleId) {
        if (TextUtils.isEmpty(articleId)) return;
        String unreadArticles = SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.UNREAD_ARTICLES));
        unreadArticles = unreadArticles.replace(articleId + ",", "");
        SpUtil.getInstance().put(getLanguageBasedKey(SpKeys.UNREAD_ARTICLES), unreadArticles);
    }

    /**
     * @return 如果本地缓存的未读文章不为空，就说明当前有未读文章
     */
    public boolean haveUnreadArticles() {
        String unreadArticles = SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.UNREAD_ARTICLES));
        return !TextUtils.isEmpty(unreadArticles);
    }

    public boolean isSectionUnread(String sectionId) {
        if (TextUtils.isEmpty(sectionId)) return false;
        List<OperateArticle> operateArticles = opArticlesMap.get(sectionId);
        if (operateArticles != null) {
            for (OperateArticle operateArticle : operateArticles) {
                if (isArticleUnread(operateArticle.getFaqMainId())) {
                    return true;
                }
            }
        }
        return false;
    }

    public boolean isArticleUnread(String articleId) {
        if (TextUtils.isEmpty(articleId)) return false;
        String unreadArticles = SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.UNREAD_ARTICLES));
        return unreadArticles.contains(articleId);
    }

    public void afterArticleEvaluated(String articleId) {
        if (TextUtils.isEmpty(articleId)) return;
        String evaluatedFaqIds = SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.EVALUATED_ARTICLES));
        SpUtil.getInstance().put(getLanguageBasedKey(SpKeys.EVALUATED_ARTICLES), String.format("%s,%s", evaluatedFaqIds, articleId));
    }

    public boolean shouldShowArticleFooter(String articleId) {
        if (TextUtils.isEmpty(articleId)) return false;
        return !SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.EVALUATED_ARTICLES)).contains(articleId);
    }

    // ******************************** 对外暴露的公共方法 ********************************


    // ******************************** 内部使用的私有方法 ********************************

    /**
     * 将所有运营文章都放到同一个集合中
     */
    private void flatRawArticleJsonArray(JSONArray rawSectionArray) {
        try {
            latelyFlatArticles = new JSONArray();
            for (int i = 0; i < rawSectionArray.length(); i++) {
                JSONObject section = JsonHelper.getJsonObject(rawSectionArray, i);
                JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
                if (faqs.length() > 0) {
                    for (int j = 0; j < faqs.length(); j++) {
                        latelyFlatArticles.put(JsonHelper.getJsonObject(faqs, j));
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    private boolean shouldIgnoreCurrentEntity(OperateSection entity) {
        if (entity == null) return true;
        for (int j = 0; j < rootSections.size(); j++) {
            OperateSection listEntity = rootSections.get(j);
            if (listEntity.getId().equals(entity.getId())) {
                return true;
            }
        }
        return false;
    }

    private void prepareRootSectionList(JSONArray rawSectionArray) {
        rootSections = new ArrayList<>();
        // 遍历所有 section，将所有有子分类的条目都挑出来，然后构建一个 父分类ID >> 子分类列表 字典
        for (int i = 0; i < rawSectionArray.length(); i++) {
            JSONObject section = JsonHelper.getJsonObject(rawSectionArray, i);
            JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
            if (faqs.length() > 0) {
                OperateSection entity = new OperateSection(
                        section.optString("sectionId"),
                        section.optString("sectionName"),
                        Integer.parseInt(section.optString("orderNo"))
                );
                if (!shouldIgnoreCurrentEntity(entity)) {
                    rootSections.add(entity);
                }
            }
        }
        Collections.sort(rootSections);
    }

    /**
     * FAQ问题列表字典：<FAQ所属分类ID, List<FaqListEntity>>
     */
    private void prepareSectionToArticlesMap(JSONArray rawSectionArray) {
        opArticlesMap = new HashMap<>();
        rawArticleMap = new HashMap<>();
        for (int i = 0; i < rawSectionArray.length(); i++) {
            JSONObject section = JsonHelper.getJsonObject(rawSectionArray, i);
            JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
            if (faqs.length() > 0) {
                String sectionId = section.optString("sectionId");

                // 运营模块分类ID >> 运营文章列表
                List<OperateArticle> faqQuestions = opArticlesMap.get(sectionId);
                if (faqQuestions == null) {
                    faqQuestions = new ArrayList<>();
                }
                for (int j = 0; j < faqs.length(); j++) {
                    JSONObject article = JsonHelper.getJsonObject(faqs, j);
                    faqQuestions.add(getOperateArticle(article));
                }
                opArticlesMap.put(sectionId, faqQuestions);

                // 分类ID >> FAQ问题列表
                rawArticleMap.put(sectionId, faqs);
            }
        }
    }

    private OperateArticle filterFaqById(JSONArray articles, String faqId) {
        if (articles != null && articles.length() > 0) {
            for (int j = 0; j < articles.length(); j++) {
                JSONObject article = JsonHelper.getJsonObject(articles, j);
                String mainId = article.optString("kmMainid");
                String contentId = article.optString("kmContentId");
                if (mainId.equals(faqId) || contentId.equals(faqId)) {
                    return getOperateArticle(article);
                }
            }
        }
        return null;
    }

    private OperateArticle getOperateArticle(JSONObject article) {
        OperateArticle articleEntity = new OperateArticle();
        articleEntity.setFaqMainId(article.optString("kmMainid"));
        articleEntity.setFaqContentId(article.optString("kmContentId"));
        articleEntity.setFaqTitle(article.optString("question"));
        articleEntity.setFaqContent(article.optString("content"));
        articleEntity.setFaqImageUrl(article.optString("imgUrl"));
        articleEntity.setFaqUpdateDate(article.optString("lastUpdateDate"));
        return articleEntity;
    }

    private String getLanguageBasedKey(String spKey) {
        return String.format("%s_%s_%s", Const.APP_ID, Const.ORIGINAL_LANGUAGE, spKey);
    }

    // ******************************** 内部使用的私有方法 ********************************

}
