package net.aihelp.data.localize;

import android.content.Context;
import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;

import net.aihelp.data.localize.config.BusinessLogicHelper;
import net.aihelp.data.localize.config.StyleSheetHelper;
import net.aihelp.data.localize.config.TextHelper;
import net.aihelp.data.localize.data.FaqHelper;
import net.aihelp.data.localize.data.LocaleStringHelper;
import net.aihelp.data.localize.data.OperateHelper;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.ui.helper.ElvaBotHelper;
import net.aihelp.utils.FileUtil;

import java.io.ByteArrayInputStream;
import java.util.Map;

import androidx.annotation.RestrictTo;


@RestrictTo(RestrictTo.Scope.LIBRARY)
public class LocalizeHelper {

    public static final int FLAG_FAQ_SECTION = 1001;
    public static final int FLAG_OP = 1002;
    public static final int FLAG_STORY = 1003;

    public static final int FLAG_LOCALE = 1005;
    public static final int FLAG_STYLE_SHEET = 1006;
    public static final int FLAG_BUSINESS_LOGIC = 1007;
    public static final int FLAG_FEEDBACK_OPINION = 1008;
    public static final int FLAG_FAQ_HOT_TOPIC = 1009;
    public static final int FLAG_PROCESS = 1010;
    public static final int FLAG_TEXT = 1011;

    public static void resetLocalizeData() {
        FaqHelper.INSTANCE.reset();
        OperateHelper.INSTANCE.reset();
        LocaleStringHelper.INSTANCE.reset();
        TextHelper.INSTANCE.reset();
    }

    public static void goFetchLocalizeData(Context context) {
        goFetchElvaBotData(context);
        if (Const.TOGGLE_LOCALIZE_VIA_INIT) {
            // 如果允许在 init 时下载，就直接走 download 逻辑就可以
            getLocalizeDataFromUrl(FLAG_FAQ_SECTION);
        } else if (LocalizeUtil.isAlreadyLocalized(FLAG_FAQ_SECTION)) {
            // 如果不允许在 init 下载，也要看一下是不是已经下好了，如果下好了，就直接准备数据源
            // 但如果本地的缓存文件是通过接口下载下来的，那就不准备数据源，等进入页面的时候拉取最新数据
            if (!LocalizeUtil.isFallbackUrl(FLAG_FAQ_SECTION, LocalizeUtil.getUrl(FLAG_FAQ_SECTION))) {
                FaqHelper.INSTANCE.prepareDataSource();
            }
        }
        getLocalizeDataFromUrl(FLAG_OP);
        getLocalizeDataFromUrl(FLAG_LOCALE);
        getLocalizeDataFromUrl(FLAG_STYLE_SHEET);
        getLocalizeDataFromUrl(FLAG_BUSINESS_LOGIC);
        getLocalizeDataFromUrl(FLAG_FAQ_HOT_TOPIC);
        getLocalizeDataFromUrl(FLAG_TEXT);
    }

    public static void goFetchElvaBotData(final Context context) {
        AIHelpRequest.getInstance().requestDownloadFile(LocalizeHelper.FLAG_STORY, new ReqCallback<String>() {
            @Override
            public void onAsyncReqSuccess(String result) {
                ElvaBotHelper.initBot(context);
            }
        });
    }

    public static void getLocalizeDataFromUrl(final int mode) {
        if (mode == FLAG_OP) {
            OperateHelper.INSTANCE.prepareHistoricDataSource();
        }
        if (mode == FLAG_FAQ_SECTION && FaqHelper.isFaqDataAlreadyPrepared()) {
            return;
        }
        AIHelpRequest.getInstance().requestDownloadFile(mode, new ReqCallback<String>() {
            @Override
            public void onAsyncReqSuccess(String result) {
                switch (mode) {
                    case FLAG_FAQ_SECTION:
                        FaqHelper.INSTANCE.prepareDataSource();
                        break;
                    case FLAG_OP:
                        OperateHelper.INSTANCE.prepareDataSource();
                        break;
                    case FLAG_LOCALE:
                        LocaleStringHelper.INSTANCE.prepareDataSource();
                        break;
                    case FLAG_STYLE_SHEET:
                        StyleSheetHelper.INSTANCE.prepareDataSource();
                        break;
                    case FLAG_BUSINESS_LOGIC:
                        BusinessLogicHelper.INSTANCE.prepareDataSource();
                        break;
                    case FLAG_FAQ_HOT_TOPIC:
                        FaqHelper.INSTANCE.prepareNotificationAndHotTopics();
                        break;
                    case FLAG_TEXT:
                        TextHelper.INSTANCE.prepareDataSource();
                        break;
                }
            }

            @Override
            public void onAsyncFailure(String url, int errorCode, String errorMsg) {
                if (mode == FLAG_FAQ_SECTION || mode == FLAG_OP) {
                    getDataAfterLocalizeFailed(mode);
                }
            }
        });
    }

    private static void getDataAfterLocalizeFailed(final int mode) {
        String url = mode == FLAG_FAQ_SECTION ? API.FAQ_URL : API.OP_URL;
        AIHelpRequest.getInstance().requestGetByAsync(url, null, new ReqCallback<String>() {
            @Override
            public void onAsyncReqSuccess(String str) {
                try {
                    if (!TextUtils.isEmpty(str)) {
                        boolean success = FileUtil.writeFileToDisk(new ByteArrayInputStream((str).getBytes()),
                                LocalizeUtil.getFileLocation(mode));
                        if (success) {
                            if (mode == FLAG_FAQ_SECTION) {
                                FaqHelper.INSTANCE.prepareDataSource();
                            } else {
                                OperateHelper.INSTANCE.prepareDataSource();
                            }
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }

            }
        });
    }

}
