package net.aihelp.core.util.permission;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;

import net.aihelp.BuildConfig;
import net.aihelp.config.AIHelpContext;

import java.lang.reflect.Method;

import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;
import androidx.fragment.app.Fragment;

public class PermissionHelper implements IPermissionCallback {

    private Activity activity;
    private Fragment fragment;
    private Object handler;
    private final String[] permissions;
    private final int requestCode;
    private int invokeId = -1;

    private PermissionHelper(Activity activity, Fragment fragment, Object handler, String[] permissions, int requestCode) {
        this.activity = activity;
        this.fragment = fragment;
        this.handler = handler;
        this.permissions = permissions;
        this.requestCode = requestCode;
        if (handler instanceof Activity) {
            this.activity = (Activity) handler;
        } else if (handler instanceof Fragment) {
            this.fragment = (Fragment) handler;
        }
    }

    public static PermissionHelper getInstance(Activity activity, Fragment fragment, Object handler, String[] permissions, int requestCode) {
        if (handler == null || permissions == null || requestCode == 0) {
            if (BuildConfig.DEBUG) {
                throw new IllegalArgumentException("handler == null || permission == null || requestCode == 0!");
            }
        }
        return new PermissionHelper(activity, fragment, handler, permissions, requestCode);
    }

    public void setInvokeId(int invokeId) {
        this.invokeId = invokeId;
    }

    public boolean avoidInvoking() {
        return invokeId == requestCode;
    }

    void invokePermissionCallback(Permission.Result result) {
        if (avoidInvoking()) return;
        if (handler != null) {
            Method[] methods = handler.getClass().getDeclaredMethods();
            for (Method method : methods) {
                Permission grantedMethod = method.getAnnotation(Permission.class);
                if (grantedMethod != null) {
                    if (requestCode == grantedMethod.requestCode()) {
                        method.setAccessible(true);
                        try {
                            method.invoke(handler, result, this);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        }
    }

    private boolean isPermissionGranted(String permission) {
        if (getActivity() != null) {
            return ContextCompat.checkSelfPermission(getActivity(), permission) == PackageManager.PERMISSION_GRANTED;
        }
        return false;
    }

    private Activity getActivity() {
        if (activity != null) {
            return activity;
        }
        if (fragment != null) {
            return fragment.getActivity();
        }
        return null;
    }

    void onRequestPermissionsResult(String[] permissions, int[] grantResults) {
        // reset the invoke id after the permission callback, so we can call the invoke method again
        setInvokeId(-1);
        for (int i = 0; i < grantResults.length; i++) {
            boolean isPermissionGranted = grantResults[i] == PackageManager.PERMISSION_GRANTED;
            if (isPermissionGranted) {
                invokePermissionCallback(Permission.Result.GRANTED);
            } else {
                if (shouldShowRequestPermissionRationale(permissions[i])) {
                    invokePermissionCallback(Permission.Result.DENIED);
                } else {
                    invokePermissionCallback(Permission.Result.GO_SETTING);
                }
            }
            // update the invoke id to requestCode again after the first loop
            // to prevent multiple callback
            setInvokeId(requestCode);
        }
    }

    Permission.State[] checkPermissionState() {
        Permission.State[] states = new Permission.State[permissions.length];
        for (int i = 0; i < permissions.length; i++) {
            String permission = permissions[i];
            Permission.State permissionState;
            if (isPermissionGranted(permission) || Build.VERSION.SDK_INT < Build.VERSION_CODES.M) {
                permissionState = Permission.State.AVAILABLE;
            } else {
                boolean hasPermissionInManifest = hasPermissionInManifest(permission);
                if (hasPermissionInManifest) {
                    permissionState = Permission.State.ASKABLE;
                    if (shouldShowRequestPermissionRationale(permission)) {
                        permissionState = Permission.State.RATIONAL;
                    }
                } else {
                    permissionState = Permission.State.UNAVAILABLE;
                }
            }
            states[i] = permissionState;
        }
        return states;
    }

    void requestPermission() {
        if (activity != null) {
            ActivityCompat.requestPermissions(activity, permissions, requestCode);
        } else if (fragment != null) {
            if (!fragment.isDetached()) {
                fragment.requestPermissions(permissions, requestCode);
            }
        }
    }

    private boolean shouldShowRequestPermissionRationale(String permission) {
        if (activity != null) {
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
                return activity.shouldShowRequestPermissionRationale(permission);
            }
        } else if (fragment != null) {
            if (!fragment.isDetached()) {
                return fragment.shouldShowRequestPermissionRationale(permission);
            }
        }
        return false;
    }

    private static boolean hasPermissionInManifest(String permission) {
        try {
            Context context = AIHelpContext.getInstance().getContext();
            PackageInfo info = context.getPackageManager().getPackageInfo(context.getPackageName(), PackageManager.GET_PERMISSIONS);
            if (info.requestedPermissions != null) {
                for (String p : info.requestedPermissions) {
                    if (p.equals(permission)) {
                        return true;
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    private void showSettingsPage() {
        Activity activity = getActivity();
        if (activity != null) {
            try {
                Intent settingsIntent = new Intent("android.settings.APPLICATION_DETAILS_SETTINGS");
                settingsIntent.addCategory("android.intent.category.DEFAULT");
                String packageName = activity.getPackageName();
                settingsIntent.setData(Uri.parse("package:" + packageName));
                activity.startActivity(settingsIntent);
            } catch (ActivityNotFoundException var3) {
                Intent i = new Intent("android.settings.MANAGE_APPLICATIONS_SETTINGS");
                i.addCategory("android.intent.category.DEFAULT");
                activity.startActivity(i);
            }
        }
    }

    public void recycle() {
        activity = null;
        fragment = null;
        handler = null;
    }

    @Override
    public void onPermissionDenied() {
        // showSettingsPage();
    }

    @Override
    public void onPermissionRational() {
        requestPermission();
    }

    @Override
    public void onPermissionIgnored() {
        showSettingsPage();
    }

}
