package net.aihelp.core.util.logger;

import android.os.Build;
import android.os.Looper;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.common.Const;
import net.aihelp.core.net.mqtt.client.MQTTException;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.bus.event.EventCenter;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.logger.controller.LoggerDBController;
import net.aihelp.data.event.FatalErrorEvent;
import net.aihelp.ui.helper.LoginMqttHelper;
import net.aihelp.utils.DeviceInfoUtil;
import net.aihelp.utils.TLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Arrays;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import androidx.annotation.NonNull;

public enum AIHelpLogger {

    INSTANCE;

    private final String TAG;
    private final LoggerDBController logStorage;
    private ThreadPoolExecutor threadPoolExecutor;

    AIHelpLogger() {
        TAG = AIHelpLogger.class.getSimpleName();
        logStorage = LoggerDBController.getInstance();

        if (threadPoolExecutor == null) {
            LinkedBlockingQueue<Runnable> logMessagesQueue = new LinkedBlockingQueue<Runnable>();
            this.threadPoolExecutor = new ThreadPoolExecutor(1, 1, 60L, TimeUnit.SECONDS, logMessagesQueue, new ThreadFactory() {
                public Thread newThread(@NonNull Runnable r) {
                    return new Thread(r, "AIHelpLogger");
                }
            });
        }
    }

    public void log(int logLevel, String message, Throwable tr) {
        this.logMessageToDatabase(logLevel, message, getStackTraceString(tr), System.currentTimeMillis());
    }

    public JSONArray getCachedLogs() {
        return logStorage.getCachedLogs();
    }

    public void deleteAllCachedLogs() {
        this.logStorage.deleteAll();
    }

    private void logMessageToDatabase(final int level, final String message, final String stacktrace, final long timeStamp) {
        try {
            threadPoolExecutor.submit(new Runnable() {
                @Override
                public void run() {
                    logStorage.insert(level, message, stacktrace, timeStamp);
                }
            });
        } catch (Exception e) {
            Log.e(this.TAG, "Rejected execution of log message : " + message, e);
        }
    }

    private static String getStackTraceString(Throwable throwable) {
        String stacktrace = "";
        if (throwable != null) {
            StringBuilder builder = new StringBuilder();
            String throwableString = Log.getStackTraceString(throwable);
            builder.append(throwableString);
            stacktrace = builder.toString();
        }
        return stacktrace;
    }

    public static final int LEVEL_FATAL = 1;
    public static final int LEVEL_ERROR = 2;
    public static final int LEVEL_WARN = 3;

    public static void fatal(String message, Throwable tr) {
        INSTANCE.log(LEVEL_FATAL, message, tr);
        if (Thread.currentThread() == Looper.getMainLooper().getThread()) {
            EventBus.getDefault().post(new FatalErrorEvent());
        }
    }

    public static void error(final String message, final Throwable tr) {
        final Matcher matcher = Pattern.compile("(?<=//).*?(?=/)").matcher(message);
        if (matcher.find()) {
            ApiExecutorFactory.getHandlerExecutor().runAsync(new Runnable() {
                @Override
                public void run() {
                    JSONObject messageObject = getMessageObject(message);
                    try {
                        InetAddress[] inetAddressArr = InetAddress.getAllByName(matcher.group());
                        if (inetAddressArr != null && inetAddressArr.length > 0) {
                            String[] ipAddressArr = new String[inetAddressArr.length];
                            for (int i = 0; i < inetAddressArr.length; i++) {
                                ipAddressArr[i] = inetAddressArr[i].getHostAddress();
                            }
                            messageObject.put("inetAddress", Arrays.toString(ipAddressArr));
                            INSTANCE.log(LEVEL_ERROR, messageObject.toString(), tr);
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                        INSTANCE.log(LEVEL_ERROR, messageObject.toString(), tr);
                    }
                }
            });
        } else {
            INSTANCE.log(LEVEL_ERROR, getMessageObject(message).toString(), tr);
        }
    }

    private static JSONObject getMessageObject(String message) {
        JSONObject msgObject = new JSONObject();
        try {
            msgObject.put("origin", message);
            msgObject.put("language", Const.ORIGINAL_LANGUAGE);
            msgObject.put("OSVersion", Build.VERSION.RELEASE);
            msgObject.put("deviceModel", Build.MODEL);
            msgObject.put("networkType", DeviceInfoUtil.getInstance().getNetworkType());
            msgObject.put("carrier", DeviceInfoUtil.getInstance().getCarrierName());
            msgObject.put("countryCode", DeviceInfoUtil.getInstance().getSimCountryIso());
            msgObject.put("proxyConfig", DeviceInfoUtil.getInstance().getProxyConfiguration());
            return msgObject;
        } catch (Exception e) {
            e.printStackTrace();
            return msgObject;
        }
    }

}

