package net.aihelp.core.util.elva;

import java.beans.PropertyChangeEvent;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Random;

import net.aihelp.core.util.elva.text.Request;
import net.aihelp.core.util.elva.text.Response;
import net.aihelp.core.util.elva.text.Sentence;
import net.aihelp.core.util.elva.text.Transformations;

import static net.aihelp.core.util.elva.text.Sentence.ASTERISK;

//import net.aihelp.core.util.elva.script.BeanshellInterpreter;

public class Context {

    private final Map<String, Object> properties = new HashMap<>();
    private final Map<String, ContextPropertyChangeListener> listeners = new HashMap<>();
    private final List<Request> requests = new LinkedList<Request>();
    private final List<Response> responses = new LinkedList<Response>();
    private final Random random = new Random();
    private long seed = 0;
    private OutputStream output;
    private Sentence that;
    private Sentence topic;
    private Transformations transformations;

    public Context() {
//        property("beanshell.interpreter", new BeanshellInterpreter());
        addContextPropertyChangeListener(new ContextRandomSeedChangeListener());
        addContextPropertyChangeListener(new ContextTopicChangeListener());
    }

    public Context(Transformations transformations) {
        this();
        this.transformations = transformations;
    }

    public void addContextPropertyChangeListener(ContextPropertyChangeListener listener) {
        listeners.put(listener.name(), listener);
    }

    public void removeContextPropertyChangeListener(ContextPropertyChangeListener listener) {
        ContextPropertyChangeListener listening = listeners.get(listener.name());
        if (listening == listener) {
            listeners.remove(listener.name());
        }
    }

    public void appendRequest(Request request) {
        requests.add(0, request);
    }

    public void appendResponse(Response response) {
        transformations.normalization(response);
        responses.add(0, response);
        that = response.lastSentence(0);
        transformations.normalization(that);
    }

    public void print(String output) throws IOException {
        outputStream().write(output.getBytes());
        outputStream().write('\n');
    }

    public Object property(String name) {
        return properties.get(name);
    }

    public void property(String name, Object value) {
        ContextPropertyChangeListener listener = listeners.get(name);
        if (listener != null) {
            Object oldValue = properties.get(name);
            PropertyChangeEvent event = new PropertyChangeEvent(this, name, oldValue, value);
            listener.propertyChange(event);
        }
        properties.put(name, value);
    }

    public OutputStream outputStream() throws IOException {
        if (output == null) {
            String path = (String) property("bot.output");
            File file = new File(path);
            if (file.isDirectory()) {
                path = file.getPath() + "/gossip-" + id() + ".txt";
            }
            outputStream(new FileOutputStream(path));
        }
        return output;
    }

    public void outputStream(OutputStream output) {
        this.output = output;
    }

    public String id() {
        String id = (String) property("bot.id");
        if ("".equals(id)) {
            return Integer.toString(hashCode());
        } else {
            return id;
        }
    }

    public Random random() {
        return random;
    }

    public void random(long seed) {
        random.setSeed(seed);
    }

    public Sentence getThat() {
        if (that == null) {
            that = ASTERISK;
        }
        return that;
    }

    public Sentence getTopic() {
        if (topic == null) {
            topic = ASTERISK;
        }
        return topic;
    }

    public void setTopic(Sentence topic) {
        if (topic == null) {
            this.topic = ASTERISK;
        }
        this.topic = topic;
    }

    public Request getRequests(int index) {
        return requests.get(index);
    }

    public Response getResponses(int index) {
        return responses.get(index);
    }

    public Transformations getTransformations() {
        return transformations;
    }

    public void setTransformations(Transformations transformations) {
        this.transformations = transformations;
    }
}