package net.aihelp.core.ui;

import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.os.Build;
import android.os.Bundle;
import android.util.DisplayMetrics;
import android.view.KeyEvent;
import android.widget.Toast;

import net.aihelp.common.Const;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.monitor.NetworkMonitorManager;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import java.lang.reflect.Method;
import java.util.Map;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.app.AppCompatActivity;
import androidx.appcompat.app.AppCompatDelegate;

public abstract class BaseActivity extends AppCompatActivity {

    protected Context mContext;

    static {
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP) {
            AppCompatDelegate.setCompatVectorFromResourcesEnabled(true);
        }
    }

    @Override
    protected void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        mContext = getBaseContext();
        if (savedInstanceState != null) {
            Intent launchIntent = AppInfoUtil.getLaunchIntent(getApplicationContext(), getPackageName());
            if (launchIntent != null) {
                finish();
                startActivity(launchIntent);
            }
            return;
        }
        setContentView(getLayoutId());
        initView();
        if (isApplyPendingTransition()) {
            overridePendingTransition(ResResolver.getAnimId("aihelp_right_in"),
                    ResResolver.getAnimId("aihelp_exit_trans"));
        }

        NetworkMonitorManager.getInstance().register(this);
    }

    @Override
    public void finish() {
        super.finish();
        if (isApplyPendingTransition()) {
            overridePendingTransition(0, ResResolver.getAnimId("aihelp_right_out"));
        }
    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        try {
            Class<?> splitCompatClass = Class.forName("com.google.android.play.core.splitcompat.SplitCompat");
            Method installActivityMethod = splitCompatClass.getDeclaredMethod("installActivity", Context.class);
            installActivityMethod.setAccessible(true);
            installActivityMethod.invoke(null, getBaseContext());
        } catch (Exception e) {

        }
        if (newConfig.fontScale != 1) getResources();
        super.onConfigurationChanged(newConfig);
    }

    @Override
    public Resources getResources() {
        Resources resources = super.getResources();
        if ("SHARP".equals(Build.MANUFACTURER)) {
            // make AIHelp's density fixed at 2.625f, which is 420dpi
            DisplayMetrics displayMetrics = resources.getDisplayMetrics();
            displayMetrics.density = 2.625f;
            displayMetrics.scaledDensity = 2.625f;
        } else if (resources.getConfiguration().fontScale != 1) {
            Configuration newConfig = new Configuration();
            newConfig.setToDefaults();
            resources.updateConfiguration(newConfig, resources.getDisplayMetrics());
        }
        return resources;
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();
        AIHelpContext.getInstance().setContext(getApplicationContext());
        NetworkMonitorManager.getInstance().unregister(this);
    }

    public void onNetworkStateChanged(NetworkState state) {
        if (state == NetworkState.NONE) {
            ToastUtil.INSTANCE.makeRawToast(mContext, ResResolver.getString("aihelp_network_no_connect"));
        }
        EventBus.getDefault().post(state);
    }

    public abstract int getLayoutId();

    public void initView() {
    }

    public boolean isApplyPendingTransition() {
        return false;
    }

    @Override
    protected void attachBaseContext(Context newBase) {
        // Update global context according to the corrected language before access resources
        Context contextWithLocale = AIHelpContext.createContextWithLocale(newBase, Const.CORRECT_LANGUAGE);
        AIHelpContext.getInstance().setContext(contextWithLocale);
        try {
            Class<?> splitCompatClass = Class.forName("com.google.android.play.core.splitcompat.SplitCompat");
            Method installActivityMethod = splitCompatClass.getDeclaredMethod("installActivity", Context.class);
            installActivityMethod.setAccessible(true);
            installActivityMethod.invoke(null, contextWithLocale);
        } catch (Exception e) {
            // ignored
        }
        super.attachBaseContext(contextWithLocale);
    }

    @Override
    public boolean onKeyDown(int keyCode, KeyEvent event) {
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            if (!Const.isNestedFragmentOnResume) return false;
        }
        return super.onKeyDown(keyCode, event);
    }

}
