package net.aihelp.config;

import android.text.TextUtils;

public class OperationConfig {

    private int selectIndex;
    private String conversationTitle;
    private ConversationConfig conversationConfig;

    public static class Builder {

        private int selectIndex = Integer.MAX_VALUE;
        private String conversationTitle;
        private ConversationConfig conversationConfig = new ConversationConfig.Builder().build();

        /**
         * set operation default selection
         *
         * Operation will select elva tab by default, you can change the default selection with this api.
         * If you set a negative index, or index larger than the total tab counts, the selected tab will still be elva.
         *
         * @param selectIndex operation default selection
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setSelectIndex(int selectIndex) {
            if (selectIndex < 0) selectIndex = Integer.MAX_VALUE;
            this.selectIndex = selectIndex;
            return this;
        }

        /**
         * Set operation support bot title, 'HELP' by default.
         *
         * @param conversationTitle operation support bot tile
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setConversationTitle(String conversationTitle) {
            if (!TextUtils.isEmpty(conversationTitle)) {
                this.conversationTitle = conversationTitle;
            }
            return this;
        }

        /**
         * Set SupportConfig through Operation
         *
         * As user can get support via Operation, you can configure support config here.
         *
         * @param conversationConfig supportConfig configure through Operation
         */
        public Builder setConversationConfig(ConversationConfig conversationConfig) {
            this.conversationConfig = conversationConfig;
            return this;
        }

        public OperationConfig build(int selectIndex, String conversationTitle, ConversationConfig conversationConfig) {
            return setSelectIndex(selectIndex).setConversationTitle(conversationTitle)
                    .setConversationConfig(conversationConfig).build();
        }

        public OperationConfig build() {
            return new OperationConfig(selectIndex, conversationTitle, conversationConfig);
        }
    }

    public int getSelectIndex() {
        return selectIndex;
    }

    public String getConversationTitle() {
        return conversationTitle;
    }

    public ConversationConfig getConversationConfig() {
        return conversationConfig;
    }

    private OperationConfig(int selectIndex, String conversationTitle, ConversationConfig conversationConfig) {
        this.selectIndex = selectIndex;
        this.conversationTitle = conversationTitle;
        this.conversationConfig = conversationConfig;
    }

    @Override
    public String toString() {
        return "OperationConfig{" +
                "selectIndex=" + selectIndex +
                ", supportBotTitle='" + conversationTitle + '\'' +
                ", supportConfig=" + conversationConfig +
                '}';
    }
}
