package net.aihelp.config;

import android.app.Activity;

import net.aihelp.config.enums.ConversationIntent;

public class ConversationConfig {

    private int conversationIntent;
    private boolean alwaysShowHumanSupportButtonInBotPage;
    private String welcomeMessage;
    private String storyNode;

    public static class Builder {
        private int conversationIntent = ConversationIntent.BOT_SUPPORT.getValue();
        private boolean alwaysShowHumanSupportButtonInBotPage;
        private String welcomeMessage;
        private String storyNode;

        /**
         * set conversation Intent for {@link net.aihelp.init.AIHelpSupport#showConversation(Activity)}
         * <p>
         * This method should show elva bot page by default.
         * If you set {@link ConversationIntent#HUMAN_SUPPORT } to this value,
         * when AIHelpSupport#showSupport is called, the sdk will show conversation page directly instead of elva bot.
         *
         * @param conversationIntent must be one of {@link ConversationIntent#BOT_SUPPORT}, {@link ConversationIntent#HUMAN_SUPPORT}
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setConversationIntent(ConversationIntent conversationIntent) {
            this.conversationIntent = conversationIntent.getValue();
            return this;
        }

        /**
         * set whether to ALWAYS show contact us button FOR ELVA BOT PAGE.
         * <p>
         * By default, user can get in touch with customer service in elva bot page only if there is already an ticket opening
         * or she/he submits a form.
         * However, if you set parameter contactUsAlwaysOnline to true, user can always see the contact us button,
         * which means they can get in touch with customer service any time.
         *
         * @param alwaysShowHumanSupportButtonInBotPage whether to ALWAYS show contact us button for elva bot page.
         */
        public Builder setAlwaysShowHumanSupportButtonInBotPage(boolean alwaysShowHumanSupportButtonInBotPage) {
            this.alwaysShowHumanSupportButtonInBotPage = alwaysShowHumanSupportButtonInBotPage;
            return this;
        }

        /**
         * set custom welcome message for customer service support
         * <p>
         * You can configure your default welcome msg in aihelp admin dashboard.(http://aihelp.net/dashboard)
         * What's more, you can show different welcome msg by different users with this api.
         *
         * @param welcomeMessage custom welcome msg
         */
        public Builder setWelcomeMessage(String welcomeMessage) {
            this.welcomeMessage = welcomeMessage;
            return this;
        }

        /**
         * set specific story node for specific scene.
         *
         * With this api call, you can show different stories in different scenes.
         * @param storyNode the story node's User Say content you configured in aihelp admin dashboard.(http://aihelp.net/dashboard)
         */
        public Builder setStoryNode(String storyNode) {
            this.storyNode = storyNode;
            return this;
        }

        public ConversationConfig build(int supportIntent, boolean contactUsAlwaysOnline, String welcomeMessage, String storyNode) {
            return new ConversationConfig(supportIntent, contactUsAlwaysOnline, welcomeMessage, storyNode);
        }

        public ConversationConfig build() {
            return new ConversationConfig(conversationIntent, alwaysShowHumanSupportButtonInBotPage, welcomeMessage, storyNode);
        }
    }

    public int getConversationIntent() {
        return conversationIntent;
    }

    public boolean isAlwaysShowHumanSupportButtonInBotPage() {
        return alwaysShowHumanSupportButtonInBotPage;
    }

    public String getWelcomeMessage() {
        return welcomeMessage;
    }

    public String getStoryNode() {
        return storyNode;
    }

    private ConversationConfig(int conversationIntent, boolean alwaysShowHumanSupportButtonInBotPage,
                               String welcomeMessage, String storyNode) {
        this.conversationIntent = conversationIntent;
        this.alwaysShowHumanSupportButtonInBotPage = alwaysShowHumanSupportButtonInBotPage;
        this.welcomeMessage = welcomeMessage;
        this.storyNode = storyNode;
    }

    @Override
    public String toString() {
        return "SupportConfig{" +
                "supportIntent=" + conversationIntent +
                ", alwaysShowHumanSupportButtonInBotPage=" + alwaysShowHumanSupportButtonInBotPage +
                ", welcomeMessage='" + welcomeMessage + '\'' +
                '}';
    }

}
