package net.aihelp.ui.cs;

import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.content.DialogInterface;
import android.os.Bundle;
import android.os.Message;
import android.text.TextUtils;
import android.util.Log;
import android.view.View;
import android.view.animation.LinearInterpolator;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;


import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.ui.dialog.AlertDialog;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.concurrent.ApiExecutor;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.event.SupportActionEvent;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.data.localize.data.FaqHelper;
import net.aihelp.data.logic.ElvaBotPresenter;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.data.model.cs.storyline.BotOrderInfo;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.data.model.cs.ElvaBotMsg;

import net.aihelp.data.model.faq.FaqContentEntity;
import net.aihelp.ui.adapter.faq.FaqAlertAdapter;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.helper.EvaluateNewHelper;
import net.aihelp.ui.wrapper.TextWatcherWrapper;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.LogoutMqttHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.helper.SendingBillHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.utils.DomainSupportHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.Styles;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import org.json.JSONArray;

import java.util.List;

import androidx.appcompat.widget.AppCompatImageView;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class ElvaBotFragment extends BaseCSFragment<ElvaBotPresenter> {

    private RecyclerView rvFaqHintList;
    private FaqAlertAdapter mFaqAlertAdapter;
    private LinearLayout llChecking;
    private AppCompatImageView ivChecking;
    private int resolveType = -1;

    public static ElvaBotFragment newInstance(Bundle bundle) {
        ElvaBotFragment elvaBot = new ElvaBotFragment();
        elvaBot.setArguments(bundle);
        return elvaBot;
    }

    @Override
    public void onResume() {
        super.onResume();

        // 运营模块从表单页面返回时无法触发当前页面的 onResume 方法，所以直接 return，在 OperationFragment 处理
        if (isOperateBot) {
            if (!MqttConfig.getInstance().isConnected()) {
                mPresenter.prepareMqttConnection(mqttCallback);
            }
            return;
        }

        // 如果是提交表单后返回的，直接跳转到人工页面
        if (ResponseMqttHelper.isFormSubmitted()) {
            EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_CONVERSATION));
        } else {
            ResponseMqttHelper.updateElvaSupportActionStatus(isAlwaysShowSupportInElva);
        }

    }

    public void onFormSubmitted() {
        // 当收到表单提交的回调时，如果已经从表单页出来了，那就会导致无法跳转人工页面
        // 因此，重写此方法并判断 isVisible，保证在收到回调后可以正常执行跳转逻辑
        if (isVisible()) {
            EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_CONVERSATION));
        }
    }

    @Override
    protected void initEventAndData(View contentView) {
        super.initEventAndData(contentView);

        StatisticHelper.whenBotVisible();

        llChecking = get("aihelp_ll_net_checking");
        ivChecking = get("aihelp_iv_checking");
        rvFaqHintList = get("aihelp_rv_faq_alert");
        rvFaqHintList.setLayoutManager(new LinearLayoutManager(getContext()));
        mFaqAlertAdapter = new FaqAlertAdapter(getContext(), new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mFaqAlertAdapter.clear();
                whenUserSay(((TextView) v).getText().toString());
                StatisticHelper.whenSendButtonClicked();
            }
        });
        rvFaqHintList.setAdapter(mFaqAlertAdapter);

        mAdapter.setOnClickedListener(new MessageListAdapter.OnClickedListenerWrapper() {

            @Override
            public void onActionClicked(String action) {
                whenUserSay(action);
            }

            @Override
            public void onUrlClicked(String url) {
                if (mPresenter.validateNetwork()) {
                    Bundle bundle = new Bundle();
                    bundle.putString(IntentValues.INTENT_URL, url);
                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                }
            }

            @Override
            public void onFormUrlClicked(long timeStamp, String formUrl) {
                if (!MqttConfig.getInstance().isConnected()) {
                    mPresenter.prepareMqttConnection(mqttCallback);
                }
                if (mPresenter.validateNetwork()) {
                    Bundle bundle = new Bundle();
                    bundle.putString(IntentValues.INTENT_URL, formUrl);
                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                    LogoutMqttHelper.updateType(LogoutMqttHelper.LOGOUT_TYPE_FORM_GOTO_PAGE);
                    StatisticHelper.whenFormEventHappened(timeStamp, StatisticHelper.FORM_ACTION_CLICKED);
                }
            }

            @Override
            public void onOrderInfoClicked(BotOrderInfo orderInfo) {
                if (mPresenter.validateNetwork()) {
                    SendingBillHelper.getInstance().readyToSelectBill(getContext(), orderInfo);
                }
            }

            @Override
            public void onFaqHelpfulClicked(boolean helpful, ElvaBotMsg botMsg) {
                if (mPresenter.validateNetwork()) {
                    mPresenter.markWhetherFaqHelpful(helpful, botMsg);
                    LogoutMqttHelper.updateType(helpful ? LogoutMqttHelper.LOGOUT_TYPE_FAQ_HELPFUL
                            : LogoutMqttHelper.LOGOUT_TYPE_FAQ_UNHELPFUL);
                }
            }

            @Override
            public void onFaqFeedbackClicked(ElvaBotMsg botMsg) {
                showAdviceAlert(botMsg);
            }

            @Override
            public void onFaqClicked(ElvaBotMsg botMsg) {
                if (mPresenter.validateNetwork()) {
                    if (botMsg != null && botMsg.getBotUrl() != null) {

                        // Try to retrieve faq mainId and contentId
                        String url = DomainSupportHelper.correctDomain(botMsg.getBotUrl().getUrlAddress());
                        int index = TextUtils.indexOf(url, "?id=");
                        if (index != -1) {
                            String faqContentId = url.substring(index + "?id=".length()).replace("&isCustom=1", "");
                            ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
                            apiExecutor.runAsync(new Runnable() {
                                @Override
                                public void run() {
                                    FaqContentEntity faqContentEntity = FaqHelper.INSTANCE.getFaqById(faqContentId);
                                    if (faqContentEntity != null) {
                                        String faqMainId = faqContentEntity.getFaqMainId();
                                        botMsg.setFaqMainId(faqMainId);
                                        botMsg.setFaqContentId(faqContentId);
                                    }
                                }
                            });
                        }

                        // initialize dialog instance, and register the dismiss callback to notify data update
                        ElvaFaqBottomSheetEvent dialog = new ElvaFaqBottomSheetEvent(ElvaBotFragment.this, botMsg);
                        dialog.setOnBotFaqInteractionCallback(new ElvaFaqBottomSheetEvent.OnBotFaqInteractionCallback() {
                            @Override
                            public void onEvaluatedFaq(boolean isHelpful) {
                                mPresenter.markWhetherFaqHelpful(isHelpful, botMsg);
                            }

                            @Override
                            public void onDialogDismissed() {
                                // TODO: 2021/11/24 把 position 传出来，只刷新对应的 item
//                                mAdapter.notifyItemChanged();
                                mAdapter.notifyDataSetChanged();
                            }
                        });
                        dialog.show(url);
                    }
                }
            }

        });

        if (!isOperateBot) {
            mPresenter.prepareMqttConnection(mqttCallback);
        }

    }

    private void showAdviceAlert(final ElvaBotMsg botMsg) {
        final AlertDialog feedbackDialog = new AlertDialog.Builder(getContext())
                .setContentView(ResResolver.getLayoutId("aihelp_dia_advice"))
                .setCancelableOntheOutside(true)
                .setWidthByDevice()
                .create();

        final EditText etFeedback = feedbackDialog.getView(ResResolver.getViewId("aihelp_et_feedback"));
        final TextView tvTitle = feedbackDialog.getView(ResResolver.getViewId("aihelp_tv_title"));
        final TextView tvCancel = feedbackDialog.getView(ResResolver.getViewId("aihelp_tv_cancel"));
        final TextView tvConfirm = feedbackDialog.getView(ResResolver.getViewId("aihelp_tv_confirm"));

        etFeedback.setHint(ResResolver.getString("aihelp_chat_hint"));
        tvTitle.setText(ResResolver.getString("aihelp_faq_feedback"));
        tvCancel.setText(ResResolver.getString("aihelp_no"));
        tvConfirm.setText(ResResolver.getString("aihelp_yes"));

        etFeedback.addTextChangedListener(new TextWatcherWrapper() {
            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                tvConfirm.setEnabled(!TextUtils.isEmpty(s.toString().trim()));
                tvConfirm.setAlpha(TextUtils.isEmpty(s.toString().trim()) ? 0.5f : 1f);
            }
        });

        feedbackDialog.setOnClickListener(ResResolver.getViewId("aihelp_tv_cancel"), new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                feedbackDialog.dismiss();
            }
        });
        feedbackDialog.setOnClickListener(ResResolver.getViewId("aihelp_tv_confirm"), new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (TextUtils.isEmpty(etFeedback.getText().toString().trim())) {
                    ToastUtil.makeRawToast(getContext(), ResResolver.getString("aihelp_faq_feedback"));
                    return;
                }
                mPresenter.postFeedbackOnFaq(botMsg, etFeedback.getText().toString().trim());
                feedbackDialog.dismiss();
                SoftInputUtil.hideSoftInput(getContext(), etFeedback);
                updateFAQFeedback(String.valueOf(botMsg.getTimeStamp()),
                        ConversationMsg.STATUS_FAQ_UNHELPFUL_FEEDBACK_GIVEN,
                        botMsg.getFaqTicketId());
            }
        });
        feedbackDialog.show();
    }

    @Override
    protected void onTextChange(final CharSequence s) {

        if (TextUtils.isEmpty(s.toString().trim())) {
            rvFaqHintList.setVisibility(View.GONE);
            mFaqAlertAdapter.clear();
            return;
        }

        final ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        apiExecutor.runAsync(new Runnable() {
            @Override
            public void run() {
                final List<FaqListEntity> faqs = mPresenter.getMatchedFaqListForAlert(String.valueOf(s));
                apiExecutor.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        rvFaqHintList.setVisibility(faqs.size() > 0 ? View.VISIBLE : View.GONE);
                        mFaqAlertAdapter.update(faqs, true);
                    }
                });
            }
        });
    }

    @Override
    protected int getLayout() {
        return ResResolver.getLayoutId("aihelp_fra_elva_bot");
    }

    @Override
    protected int getLoadingTargetViewId() {
        return ResResolver.getViewId("aihelp_elva_root");
    }

    @Override
    public void onClick(View v) {
        if (v.getId() == ResResolver.getViewId("aihelp_btn_send")) {
            StatisticHelper.whenSendButtonClicked();
            String input = etInput.getText().toString();
            whenUserSay(input);
        }
    }

    private void whenUserSay(String userSay) {
        updateChatList(ConversationHelper.getUserTextMsg(true, userSay));
        Message message = Message.obtain();
        message.obj = userSay;
        mHandler.sendMessage(message);
    }

    @Override
    protected void handleMsg(Message msg) {

        String userInput = (String) msg.obj;
        final ElvaBotMsg botReply = ConversationHelper.getBotReply(userInput);
        if (!botReply.isBotStupid() || !mPresenter.isNetworkAvailable()) {
            rvMsgList.postDelayed(new Runnable() {
                @Override
                public void run() {
                    updateChatList(botReply);
                }
            }, 800);
        } else {
            mPresenter.askForAnswer(userInput);
        }

    }

    @Override
    public void updateChatList(ConversationMsg msg) {
        super.updateChatList(msg);

        // 对每组机器人问答进行打点统计，同时更新标签信息，用于同步机器人客诉
        if (msg instanceof ElvaBotMsg) {
            ElvaBotMsg botMsg = (ElvaBotMsg) msg;
            mPresenter.handleBotMsg(botMsg);
        }

    }

    public void updateFAQFeedback(String timeMillis, boolean isLike, String ticketId) {
        for (int i = 0; i < mAdapter.getDataList().size(); i++) {
            ConversationMsg conversationMsg = mAdapter.getDataList().get(i);
            if (conversationMsg.getMsgType() == ConversationMsg.TYPE_ADMIN_FAQ) {
                if (String.valueOf(conversationMsg.getTimeStamp()).equals(timeMillis)) {
                    conversationMsg.setMsgStatus(isLike ? ConversationMsg.STATUS_FAQ_HELPFUL : ConversationMsg.STATUS_FAQ_UNHELPFUL);
                    conversationMsg.setFaqTicketId(ticketId);
                    mAdapter.notifyItemChanged(i);
                }
            }
        }
    }

    public void updateFAQFeedback(String timeMillis, int msgStatus, String ticketId) {
        for (int i = 0; i < mAdapter.getDataList().size(); i++) {
            ConversationMsg conversationMsg = mAdapter.getDataList().get(i);
            if (conversationMsg.getMsgType() == ConversationMsg.TYPE_ADMIN_FAQ) {
                if (String.valueOf(conversationMsg.getTimeStamp()).equals(timeMillis)) {
                    conversationMsg.setMsgStatus(msgStatus);
                    conversationMsg.setFaqTicketId(ticketId);
                    mAdapter.notifyItemChanged(i);
                }
            }
        }
    }

    @Override
    public void onMqttLogin(List<ConversationMsg> mqttReplyMsg) {
        super.onMqttLogin(mqttReplyMsg);
        if (isVisible() && (ResponseMqttHelper.isTicketActive() || ResponseMqttHelper.isTicketRejected())) {
            EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_CONVERSATION));
        }
    }

    public void markSupportActionUnread() {
        // flag for previous page to show unread status
        ResponseMqttHelper.setHasUnreadMsg(true);

        // To avoid showing CS entrance while filling form when there is a response
        if (isVisible()) {
            EventBus.getDefault().post(new SupportActionEvent(IntentValues.SHOW_SUPPORT_ACTION,
                    IntentValues.SUPPORT_ACTION_MSG_UNREAD));
        }
    }

    @Override
    public void updateNetCheckingStatus(boolean startChecking) {
        if (llChecking != null && ivChecking != null) {
            if (startChecking) {
                TextView tvNetChecking = get("aihelp_tv_net_checking");
                tvNetChecking.setText(ResResolver.getString("aihelp_network_checking"));
                llChecking.setVisibility(View.VISIBLE);
                ObjectAnimator loadingAnimator = ObjectAnimator.ofFloat(ivChecking, "rotation", 0F, 360F);
                loadingAnimator.setDuration(1000);
                loadingAnimator.setInterpolator(new LinearInterpolator());
                loadingAnimator.setRepeatCount(ValueAnimator.INFINITE);
                loadingAnimator.start();
            } else {
                llChecking.setVisibility(View.GONE);
            }
        }
    }

    @Override
    public boolean isNetCheckingInProgress() {
        return llChecking != null && llChecking.getVisibility() == View.VISIBLE;
    }

    public boolean onBackPressed() {
        if (isNetCheckingInProgress()) {
            showNetworkInterruptDialog();
            return false;
        }
        // 解决状态调查前提：当前语言的开关打开 && 当前没有进行中客诉 && 与机器人发生过交互
        if (CustomConfig.CustomerService.isBotResolveQueryEnable &&
                !ResponseMqttHelper.isTicketActive() &&
                mAdapter != null && mAdapter.getItemCount() > 1) {
            EvaluateNewHelper.getInstance().askAboutIssueResolvedStatus(getContext(), null, new EvaluateNewHelper.OnConfirmResolveStatusListener() {
                @Override
                public void onConfirmResolve(boolean isResolved) {
                    resolveType = isResolved ? 1 : 2;
                    CustomConfig.CustomerService.isBotResolveQueryEnable = false;
                    if (getActivity() != null) getActivity().onBackPressed();
                }
            });
            return false;
        }
        return true;
    }

    private void showNetworkInterruptDialog() {
        if (getActivity() != null) {
            final AlertDialog netCheckDialog = new AlertDialog.Builder(getContext())
                    .setContentView(ResResolver.getLayoutId("aihelp_dia_interrupt_net_check"))
                    .setWidthByDevice()
                    .create();

            final TextView tvTitle = netCheckDialog.getView(ResResolver.getViewId("aihelp_tv_title"));
            final TextView tvCancel = netCheckDialog.getView(ResResolver.getViewId("aihelp_tv_cancel"));
            final TextView tvConfirm = netCheckDialog.getView(ResResolver.getViewId("aihelp_tv_confirm"));

            tvTitle.setText(ResResolver.getString("aihelp_network_check_in_progress"));
            tvCancel.setText(ResResolver.getString("aihelp_no"));
            tvConfirm.setText(ResResolver.getString("aihelp_yes"));

            netCheckDialog.setOnClickListener(ResResolver.getViewId("aihelp_tv_cancel"), new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    netCheckDialog.dismiss();
                }
            });
            netCheckDialog.setOnClickListener(ResResolver.getViewId("aihelp_tv_confirm"), new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    netCheckDialog.dismiss();
                    if (llChecking != null) llChecking.setVisibility(View.GONE);
                    if (getActivity() != null) getActivity().onBackPressed();
                }
            });

            if (getActivity() != null && !getActivity().isFinishing()) {
                netCheckDialog.show();
            } else {
                TLog.e("Activity finished when network dialog is about to show.");
            }
        }

    }

    @Override
    public void onNetworkStateChanged(NetworkState networkState) {
        if (networkState == NetworkState.NONE) {
            mqttCallback.onMqttFailure(null);
        } else {
            mPresenter.prepareMqttConnection(mqttCallback);
        }
    }

    @Override
    public void onDestroy() {
        if (mPresenter != null) mPresenter.logoutMqtt(false, resolveType);
        super.onDestroy();
    }

}
