package net.aihelp.ui.cs;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Intent;
import android.graphics.Color;
import android.os.Build;
import android.os.Bundle;
import android.os.Message;
import android.text.TextUtils;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.common.UserProfile;
import net.aihelp.core.net.monitor.NetworkMonitor;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.luban.Luban;
import net.aihelp.core.util.luban.OnCompressListener;
import net.aihelp.core.util.permission.AIHelpPermissions;
import net.aihelp.core.util.permission.IPermissionCallback;
import net.aihelp.core.util.permission.Permission;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.data.event.PrepareMessageTimeStampEvent;
import net.aihelp.data.logic.ConversationPresenter;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.helper.AttachmentPicker;
import net.aihelp.ui.helper.AttachmentPickerFile;
import net.aihelp.ui.helper.BitmapHelper;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.EvaluateNewHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.ui.widget.AIHelpSwipeRefreshLayout;
import net.aihelp.ui.widget.snackbar.Snackbar;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.MediaUtils;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.Styles;
import net.aihelp.utils.ToastUtil;

import org.json.JSONArray;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Pattern;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.AppCompatButton;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class ConversationFragment extends BaseCSFragment<ConversationPresenter> implements AttachmentPicker.AttachmentPickerListener {

    private boolean isModeConversation;
    private AIHelpSwipeRefreshLayout mRefreshLayout;
    private int mCurrentPage = 1;
    private boolean isTicketAlreadyFinished;

    private LinearLayout llInput;
    private LinearLayout llRate;

    public static ConversationFragment newInstance(Bundle bundle) {
        ConversationFragment fraConversation = new ConversationFragment();
        fraConversation.setArguments(bundle);
        return fraConversation;
    }

    @Override
    protected void initEventAndData(View contentView) {
        super.initEventAndData(contentView);

        llInput = get("aihelp_ll_input");
        llRate = get("aihelp_ll_rate");
        mRefreshLayout = get("aihelp_refresh_layout");
        etInput.setText(mPresenter.getInputDraft());

        TextView rateInvite = get("aihelp_tv_invite_rate");
        Styles.reRenderTextView(rateInvite, CustomConfig.CustomerService.csInviteEvaluate);

        AppCompatButton rateButton = get("aihelp_btn_go_rate");
        rateButton.setText(ResResolver.getString("aihelp_rate_button"));
        rateButton.setBackground(Styles.getDrawable(Color.parseColor(CustomConfig.CommonSetting.interactElementTextColor), 8));

        mAdapter.setOnClickedListener(new MessageListAdapter.OnClickedListenerWrapper() {

            @Override
            public void onUrlClicked(String url) {
                if (mPresenter.validateNetwork()) {
                    Bundle bundle = new Bundle();
                    bundle.putString(IntentValues.INTENT_URL, url);
                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                }
            }

            @Override
            public void onRetrySendingMessage(int position, ConversationMsg msg) {

                if (!mPresenter.isNetworkAvailable() || !MqttConfig.getInstance().isConnected()) {
                    ToastUtil.makeRawToast(getContext(), ResResolver.getString("aihelp_network_no_connect"));
                    return;
                }

                if (isTicketAlreadyFinished) {
                    ToastUtil.makeRawToast(getContext(), ResResolver.getString("aihelp_ticket_closed"));
                    return;
                }

                mAdapter.remove(position);
                mAdapter.notifyItemChanged(position);

                if (msg.getMsgType() == ConversationMsg.TYPE_USER_TEXT) {
                    msg.setTimeStamp(System.currentTimeMillis());
                    updateChatList(msg);
                    chatWithSupportWithTimeout(msg.getTimeStamp(), msg.getMsgContent());
                }

                if (msg.getMsgType() == ConversationMsg.TYPE_USER_IMAGE) {
                    uploadImage(msg.getMsgContent());
                }

                if (msg.getMsgType() == ConversationMsg.TYPE_USER_VIDEO) {
                    uploadVideo(msg.getMsgContent());
                }

            }

        });

        mRefreshLayout.setEnabled(CustomConfig.CustomerService.isHistoryChatEnable);
        mRefreshLayout.setColorSchemeColors(Styles.getColor(CustomConfig.CommonSetting.interactElementTextColor));
        if (CustomConfig.CustomerService.isHistoryChatEnable) {
            mRefreshLayout.setOnRefreshListener(new AIHelpSwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    mPresenter.getLastConversation(mCurrentPage);
                }
            });
        }

        ImageView ivAttach = get("aihelp_iv_attach");
        ivAttach.setVisibility(View.VISIBLE);
        ivAttach.setOnClickListener(this);
        Styles.reRenderImageView(ivAttach, "aihelp_svg_ic_attach");

        get("aihelp_btn_go_rate").setOnClickListener(this);
        ResponseMqttHelper.setHasUnreadMsg(false);

        if (!isOperateHuman) {
            mPresenter.prepareMqttConnection(mqttCallback);
        }

    }

    @Override
    protected void getBundleAfterDataPrepared(Bundle extras) {
        isModeConversation = IntentValues.MODE_CONVERSATION == extras.getInt(IntentValues.SUPPORT_MODE);
    }

    @Override
    public void onResume() {
        super.onResume();

        if (!MqttConfig.getInstance().isConnected() && isOperateHuman) {
            mPresenter.prepareMqttConnection(mqttCallback);
            return;
        }

        // After form submitted, scroll the message list to the end to display form content
        if (ResponseMqttHelper.isFormSubmitted()) {
            ResponseMqttHelper.setFormSubmitStatus(false);
            rvMsgList.scrollToPosition(mAdapter.getItemCount() - 1);
        }

    }

    @Override
    public void onDestroy() {
        if (mPresenter != null && mAdapter != null) {
            mPresenter.saveInputDraftIfNeeded(etInput.getText().toString().trim());
            if (isOperateHuman || isModeConversation) mPresenter.logoutMqtt(true);
        }
        super.onDestroy();
    }

    @Override
    protected int getLayout() {
        return ResResolver.getLayoutId("aihelp_fra_conversation");
    }

    @Override
    protected int getLoadingTargetViewId() {
        return ResResolver.getViewId("aihelp_conversation_root");
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        AIHelpPermissions.getInstance().onRequestPermissionsResult(grantResults.length == 1 && grantResults[0] == 0);
    }

    @Permission(requestCode = 1000)
    public void launchPicker(Permission.Result result, final IPermissionCallback callback) {
        switch (result) {
            case GRANTED:
                AttachmentPicker.getInstance().setPickerHost(this).launchImagePicker(true);
                break;
            case DENIED:
                ToastUtil.showRawSnackBar(getActivity(), ResResolver.getString("aihelp_permission_denied"), Snackbar.LENGTH_LONG);
                break;
            case RATIONAL:
                ToastUtil.showRawSnackBar(getActivity(), ResResolver.getString("aihelp_permission_denied"),
                        ResResolver.getString("aihelp_yes"), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                callback.onPermissionRational();
                            }
                        });
                break;
            case GO_SETTING:
                ToastUtil.showRawSnackBar(getActivity(), ResResolver.getString("aihelp_permission_ignored"),
                        ResResolver.getString("aihelp_permission_settings"), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                callback.onPermissionIgnored();
                            }
                        });
                break;
            case NONE:
                AttachmentPicker.getInstance().setPickerHost(this).launchImagePicker(false);
                break;
        }
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent intent) {
        super.onActivityResult(requestCode, resultCode, intent);
        if (intent != null && resultCode == Activity.RESULT_OK) {
            AttachmentPicker.getInstance().onAttachmentPickRequestResult(requestCode, intent);
        }
    }

    private void chatWithSupportWithTimeout(long timeStamp, String input) {
        mPresenter.chatWithSupport(timeStamp, input);
        sendDelayTimeoutMsg(timeStamp, 20);
    }

    public void onMqttLogin(List<ConversationMsg> mqttReplyMsg) {
        if (mAdapter.getItemCount() > 0) {
            List<ConversationMsg> exceptionList = getListIfDisconnectionEverHappened();
            if (!ListUtil.isListEmpty(exceptionList)) {
                mqttReplyMsg.addAll(exceptionList);
                Collections.sort(mqttReplyMsg, new Comparator<ConversationMsg>() {
                    @Override
                    public int compare(ConversationMsg o1, ConversationMsg o2) {
                        Long o1Time = o1.getTimeStamp();
                        Long o2Time = o2.getTimeStamp();
                        return o1Time.compareTo(o2Time);
                    }
                });
                for (ConversationMsg msg : exceptionList) {
                    if (msg.getMsgStatus() == ConversationMsg.STATUS_SENDING) {
                        if (msg.getMsgType() == ConversationMsg.TYPE_USER_IMAGE) {
                            mPresenter.uploadImage(new File(msg.getMsgContent()), msg.getTimeStamp());
                        } else if (msg.getMsgType() == ConversationMsg.TYPE_USER_VIDEO) {
                            mPresenter.uploadVideo(new File(msg.getMsgContent()), msg.getTimeStamp());
                        } else {
                            chatWithSupportWithTimeout(msg.getTimeStamp(), msg.getMsgContent());
                        }
                    }
                }
            }
        }
        // reset current page, in case of reconnecting after network disconnect.
        mCurrentPage = 1;
        super.onMqttLogin(mqttReplyMsg);

        // do not show rating by default, show button instead.
        // changed at 2021/07/12 since v2.5.0.
        if (isVisible() && (ResponseMqttHelper.isTicketRejected() || ResponseMqttHelper.isTicketFinished())) {
            hideInputAfterConversationFinished();
        }
    }

    private List<ConversationMsg> getListIfDisconnectionEverHappened() {
        List<ConversationMsg> failedList = new ArrayList<>();
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            ConversationMsg msg = mAdapter.getDataList().get(i);
            if (msg.getMsgStatus() != ConversationMsg.STATUS_SUCCESS) {
                failedList.add(0, msg);
            }
        }
        return failedList;
    }

    public void hideInputAfterConversationFinished() {
        // if the ticket is finished when user's selecting pictures or videos,
        // he will just generate a brand new ticket by uploading them.
        // so mark the ticket's finish status, in which occasion the user's final select result will be omitted.
        isTicketAlreadyFinished = true;
        // hide input area and display the rate button when asking-resolve-status-toggle is open or asking-for-rating-toggle is open
        llInput.postDelayed(new Runnable() {
            @Override
            public void run() {
                boolean ticketEvaluable = ResponseMqttHelper.isTicketWaitForAskingResolveStatus() || ResponseMqttHelper.isTicketWaitForRating();
                if (ticketEvaluable && !ResponseMqttHelper.isTicketRejected()) {
                    llInput.setVisibility(View.GONE);
                    llRate.setVisibility(View.VISIBLE);
                } else {
                    llInput.setVisibility(View.GONE);
                    llRate.setVisibility(View.GONE);
                }
            }
        }, 0);
        SoftInputUtil.hideSoftInput(getContext(), llInput);
    }

    @Override
    public void onPickSuccess(AttachmentPickerFile file, Bundle bundle) {
        if (isTicketAlreadyFinished) return;

        String path = file.filePath;
        if (TextUtils.isEmpty(path)) {
            ToastUtil.makeText(getContext(), "Failed to get file path.", false);
            return;
        }

        if (Pattern.compile("^/.+\\.(png|jpg|jpeg|PNG|JPG|JPEG)$").matcher(path).matches()) {
            uploadImage(path);
        }

        if (Pattern.compile("^/.+\\.(mp4|MP4)$").matcher(path).matches()) {
            uploadVideo(path);
        }

    }

    private void uploadVideo(final String path) {
        MediaUtils.getImageForVideo(path, new MediaUtils.OnLoadVideoImageListener() {
            @Override
            public void onLoadImage(File file) {
                final ConversationMsg videoMsg = new ConversationMsg(ConversationMsg.TYPE_USER_VIDEO, ConversationMsg.STATUS_SENDING);
                videoMsg.setMsgContent(path);
                updateChatList(videoMsg);
                mPresenter.uploadVideo(new File(path), videoMsg.getTimeStamp());
                sendDelayTimeoutMsg(videoMsg.getTimeStamp(), 150);
            }
        });

    }

    private void uploadImage(final String path) {
        Luban.with(getContext()).load(path).setCompressListener(new OnCompressListener() {
            @Override
            public void onStart() {
            }

            @Override
            public void onSuccess(File compressedFile) {
                uploadAfterCompress(compressedFile);
            }

            @Override
            public void onError(Throwable e) {
                uploadAfterCompress(new File(path));
            }

            private void uploadAfterCompress(File file) {
                ConversationMsg imageMsg = new ConversationMsg(ConversationMsg.TYPE_USER_IMAGE, ConversationMsg.STATUS_SENDING);
                imageMsg.setImageSize(BitmapHelper.computeSize(file.getPath()));
                imageMsg.setMsgContent(file.getPath());
                updateChatList(imageMsg);
                mPresenter.uploadImage(file, imageMsg.getTimeStamp());
                sendDelayTimeoutMsg(imageMsg.getTimeStamp(), 50);
            }

        }).launch();
    }

    @SuppressLint("StringFormatInvalid")
    @Override
    public void onPickFailure(int code, Long limitSize) {
        switch (code) {
            case AttachmentPicker.ATTACHMENT_FILE_SIZE_LIMIT_EXCEEDED:
                String msg = ResResolver.getString("aihelp_media_upload_err_size");
                try {
                    ToastUtil.makeRawToast(getContext(), String.format(msg, String.valueOf(limitSize / 1048576)));
                } catch (Exception e) {
                    e.printStackTrace();
                    ToastUtil.makeRawToast(getContext(), msg + ", < " + (limitSize / 1048576) + "M");
                }
                break;
            case AttachmentPicker.INVALID_URI:
                ToastUtil.makeRawToast(getContext(), "Failed to get specific resource");
                break;
        }
    }

    public void withdrawMsg(long withdrawTimeStamp) {
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            ConversationMsg msg = mAdapter.getDataList().get(i);
            if (msg.getTimeStamp() == withdrawTimeStamp) {
                mAdapter.getDataList().remove(i);
                mAdapter.notifyDataSetChanged();
                mPresenter.updateCachedUnreadMessageCount(true);
                break;
            }
        }
    }

    @NetworkMonitor
    public void onNetworkStateChanged(NetworkState networkState) {
        if (networkState == NetworkState.NONE) {
            mqttCallback.onMqttFailure(null);
        } else {
            // If this page comes from bot, the mqtt connect should be requested by that page, not here.
            if (isModeConversation) {
                mPresenter.prepareMqttConnection(mqttCallback);
            }
        }
    }

    public void onLastConversationRetrieved(List<ConversationMsg> msgList) {
        mRefreshLayout.setRefreshing(false);
        if (msgList != null) {
            mCurrentPage++;
            mAdapter.insertHistoryConversation(msgList);
            RecyclerView.LayoutManager layoutManager = rvMsgList.getLayoutManager();
            if (layoutManager instanceof LinearLayoutManager) {
                LinearLayoutManager linearManager = (LinearLayoutManager) layoutManager;
                int lastItemPosition = linearManager.findLastVisibleItemPosition();
                rvMsgList.scrollToPosition(msgList.size() - 1 + lastItemPosition);
                rvMsgList.smoothScrollBy(0, -100);
            }
        }
    }

    public void onFormSubmitted(String formContent) {
        mAdapter.update(ConversationHelper.getUserFormMsgList(formContent), false);
        rvMsgList.scrollToPosition(mAdapter.getItemCount() - 1);
    }

    private void sendDelayTimeoutMsg(long timeStamp, long delaySeconds) {
        Message message = Message.obtain();
        message.obj = timeStamp;
        message.what = getMessageWhatFromTimeStamp(String.valueOf(timeStamp));
        mHandler.sendMessageDelayed(message, delaySeconds * 1000);
    }

    protected void handleMsg(Message msg) {
        updateMsgStatus(false, (Long) msg.obj);
    }

    private int getMessageWhatFromTimeStamp(String timeStamp) {
        return Integer.parseInt(timeStamp.substring(timeStamp.length() / 5 * 2));
    }

    public void updateMsgStatus(boolean isSuccess, long timeStamp) {
        mHandler.removeMessages(getMessageWhatFromTimeStamp(String.valueOf(timeStamp)));
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            ConversationMsg msg = mAdapter.getDataList().get(i);
            if (msg.getTimeStamp() == timeStamp) {
                msg.setMsgStatus(isSuccess ? ConversationMsg.STATUS_SUCCESS : ConversationMsg.STATUS_RETRY);
                mAdapter.notifyItemChanged(i);
                break;
            }
        }
    }

    @Override
    public void onEventComing(PrepareMessageTimeStampEvent event) {
        super.onEventComing(event);
        mPresenter.updateCachedUnreadMessageCount(false);
    }

    @Override
    public void onClick(View v) {
        if (v.getId() == ResResolver.getViewId("aihelp_btn_send")) {
            if (mPresenter.validateNetwork()) {
                StatisticHelper.whenSendButtonClicked();
                String input = etInput.getText().toString().trim();
                mPresenter.clearInputDraft();
                ConversationMsg msg = ConversationHelper.getUserTextMsg(false, input);
                updateChatList(msg);
                chatWithSupportWithTimeout(msg.getTimeStamp(), input);
            }
        }

        if (v.getId() == ResResolver.getViewId("aihelp_iv_attach")) {
            if (mPresenter.validateNetwork()) {
                AIHelpPermissions.getInstance().setHost(ConversationFragment.this).setRequestCode(1000)
                        .setRequestPermission(Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN ?
                                Manifest.permission.READ_EXTERNAL_STORAGE : "")
                        .request();
            }
        }

        if (v.getId() == ResResolver.getViewId("aihelp_btn_go_rate")) {
            EvaluateNewHelper.getInstance().show(getContext(), new EvaluateNewHelper.OnConfirmEvaluateListener() {
                @Override
                public boolean onConfirmEvaluate(int rate, String input, JSONArray selected) {
                    if (!mPresenter.isNetworkAvailable() && !MqttConfig.getInstance().isConnected()) {
                        Toast.makeText(getContext(), ResResolver.getString("aihelp_network_no_connect"), Toast.LENGTH_SHORT).show();
                        return false;
                    }
                    bottomLayout.setVisibility(View.GONE);
                    if (ResponseMqttHelper.isTicketWaitForRating()) {
                        mPresenter.evaluateSupport(rate, input, selected);
                    }
                    // reset current user's unread message count to 0 after the evaluation is finished
                    SpUtil.getInstance().put(UserProfile.USER_ID, 0);
                    // reset all flags after the ticket is finished to make sure that after finished a ticket,
                    // and go back to bot page, there is no human support entrance at the top-right corner
                    ResponseMqttHelper.resetTicketStatusFlags();
                    mRefreshLayout.setEnabled(false);
                    return true;
                }
            });
        }
    }

}
