package net.aihelp.data.logic;

import net.aihelp.R;
import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.common.CustomConfig;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.mqtt.AIHelpMqtt;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.data.event.LoadingElvaEvent;
import net.aihelp.data.event.PrepareMessageTimeStampEvent;
import net.aihelp.data.event.SupportActionEvent;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.ui.cs.BaseCSFragment;
import net.aihelp.ui.cs.ConversationFragment;
import net.aihelp.ui.cs.ElvaBotFragment;
import net.aihelp.ui.faq.BaseFaqFragment;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.ElvaBotHelper;
import net.aihelp.ui.helper.LoginMqttHelper;
import net.aihelp.ui.helper.LogoutMqttHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.utils.TLog;

import org.json.JSONObject;

import java.util.List;

import androidx.fragment.app.Fragment;


public class MqttCallbackImpl implements IMqttCallback {

    private BaseFaqFragment faqFragment;
    private BaseCSFragment csFragment;
    private ElvaBotFragment csElvaBot;
    private ConversationFragment csConversation;

    public void updateHostView(Fragment fragment) {

        if (fragment instanceof BaseCSFragment) {
            csFragment = (BaseCSFragment) fragment;
            if (fragment instanceof ElvaBotFragment) {
                csElvaBot = (ElvaBotFragment) fragment;
            } else if (fragment instanceof ConversationFragment) {
                csConversation = (ConversationFragment) fragment;
            }
        } else if (fragment instanceof BaseFaqFragment) {
            faqFragment = (BaseFaqFragment) fragment;
        }

    }

    @Override
    public void onMqttConnected() {

    }

    @Override
    public void onMqttResponse(int loginType, String topic, String response) {
        try {

            JSONObject responseObject = new JSONObject(response);
            boolean isCodeNot200 = responseObject.has("code") && responseObject.optInt("code") != 200;
            boolean hasErrorCode = responseObject.has("errorCode");
            if (isCodeNot200 || hasErrorCode) return;

            switch (topic) {
                // login
                case API.TOPIC_LOGIN:
                    dismissMqttLoading();
                    List<ConversationMsg> mqttReplyMsg = LoginMqttHelper.getLoginResponse(response);
                    csFragment.onMqttLogin(mqttReplyMsg);
                    break;
                case API.TOPIC_LOGOUT:
                    AIHelpMqtt.getInstance().logoutMqttConnection();
                    break;

                // 机器人客诉相关
                case API.TOPIC_BOT_CHAT:
                    if (csElvaBot != null) {
                        csElvaBot.updateChatList(ElvaBotHelper.getMqttReply(response));
                    }
                    break;
                case API.TOPIC_BOT_FAQ:
                    if (csElvaBot != null) {
                        String timeMillis = responseObject.optString("timeMillis");
                        String isLike = responseObject.optString("isLike");
                        String ticketId = responseObject.optString("ticketId");
                        int msgStatus = "1".equals(isLike) ? ConversationMsg.STATUS_FAQ_HELPFUL : ConversationMsg.STATUS_FAQ_UNHELPFUL;
                        csElvaBot.updateFAQFeedback(timeMillis, msgStatus, ticketId);
                    }
                    break;

                // 人工客诉相关
                case API.TOPIC_CONVERSATION_SEND:
                    ResponseMqttHelper.setTicketActive(true);
                    if (csConversation != null) {
                        boolean postMsgSuccess = "ok".equals(responseObject.optString("state"));
                        csConversation.updateMsgStatus(postMsgSuccess, responseObject.optLong("timeStamp"));
                        if (postMsgSuccess) {
                            ResponseMqttHelper.tryUploadLog(responseObject.optBoolean("isUploadLog"));
                        }
                    }
                    break;
                case API.TOPIC_WITHDRAW:
                    if (csConversation != null) {
                        csConversation.withdrawMsg(ConversationHelper.getWithdrawTimeStamp(response));
                    }
                    break;
                case API.TOPIC_CONVERSATION_RECEIVE:
                    if (csConversation != null && csConversation.isVisible()) {
                        ConversationMsg replyMsg = ConversationHelper.getSupportReplyMsg(response);
                        EventBus.getDefault().post(new PrepareMessageTimeStampEvent(replyMsg));
                        csConversation.updateChatList(replyMsg);
                        return;
                    }

                    if (csElvaBot != null) {
                        csElvaBot.markSupportActionUnread();
                    }
                    break;
                case API.TOPIC_CONVERSATION_FINISHED:
                    // 是否展示解决、未解决弹窗
                    ResponseMqttHelper.setTicketWaitForAskingResolveStatus(responseObject.optBoolean("isShowResolve"));
                    // 是否展示给客服评分弹窗，对于新客诉来说，需要在客诉结束时判断配置开关的状态
                    ResponseMqttHelper.setTicketWaitForRating(CustomConfig.CustomerService.isTicketRatingEnable);
                    // 是否展示 「去评分」 按钮
                    ResponseMqttHelper.setAppRatable("yes".equals(responseObject.optString("storeReview")));
                    if (csConversation != null && csConversation.isVisible()) {
                        csConversation.hideInputAfterConversationFinished();
                        return;
                    }
                    if (csElvaBot != null) {
                        csElvaBot.markSupportActionUnread();
                    }
                    break;
                case API.TOPIC_TICKET_REJECTED:
                    if (csConversation != null && csConversation.isVisible()) {
                        ResponseMqttHelper.setTicketRejected(true);
                        csConversation.hideInputAfterConversationFinished();
                    }
                    break;
                case API.TOPIC_CONVERSATION_EVALUATE:
                    break;
                case API.TOPIC_SUBMIT_FORM:

                    // when user submit a form, setFormSubmitStatus to sync form message to list,
                    // and setTicketActive & setHasUnreadMsg to make FAQ list be same as bot page.
                    // All these flags will be reset when user visited human support page.
                    ResponseMqttHelper.setFormSubmitStatus(true);
                    ResponseMqttHelper.setTicketActive(true);

                    // 将填写的表单信息插入到人工消息列表中
                    if (csConversation != null) {
                        String formContent = responseObject.optString("msg");
                        csConversation.onFormSubmitted(formContent);
                    }

                    // 通知机器人页面表单已提交，不再需要标记未读消息，因为会直接跳转人工
                    // Since v3.1.2
                    if (csElvaBot != null) {
                        csElvaBot.onFormSubmitted();
                    }

                    LogoutMqttHelper.updateType(LogoutMqttHelper.LOGOUT_TYPE_FORM_SUBMIT);
                    StatisticHelper.whenFormEventHappened(StatisticHelper.getClickedFormTimeStamp(), StatisticHelper.FORM_ACTION_SUBMITTED);
                    ResponseMqttHelper.tryUploadLog(responseObject.getBoolean("isUploadLog"));

                    if (Const.sSpecificFormSubmittedListener != null && responseObject.getBoolean("isSpecificForm")) {
                        Const.sSpecificFormSubmittedListener.onFormSubmitted();
                    }
                    break;
                case API.TOPIC_SUBMIT_ORDER:
                    ResponseMqttHelper.setTicketActive(true);
                    ResponseMqttHelper.setHasUnreadMsg(true);
                    // If the sending bill dialog is visible and network is reconnect,
                    // the page maybe jumping to manual support page
                    // In this case, there should never show an unread status
                    if (csElvaBot != null && csElvaBot.isVisible()) {
                        EventBus.getDefault().post(new SupportActionEvent(IntentValues.SHOW_SUPPORT_ACTION,
                                IntentValues.SUPPORT_ACTION_MSG_UNREAD));
                    }
                    break;

                // FAQ 未读提醒
                case API.TOPIC_FAQ_NOTIFICATION:
                    if (faqFragment != null) {
                        faqFragment.showEntranceWithNotification(true);
                    }
                    break;
            }
        } catch (Exception e) {
            TLog.e("onMqttResponse, Exception " + e.toString());
            AIHelpLogger.error(response, e);
        }
    }

    @Override
    public void onMqttFailure(String message) {
        showMqttLoading();
    }

    @Override
    public void onMqttException() {

    }

    @Override
    public void showMqttLoading() {
        EventBus.getDefault().post(new LoadingElvaEvent(IntentValues.SHOW_CS_LOADING));
    }

    @Override
    public void dismissMqttLoading() {
        EventBus.getDefault().post(new LoadingElvaEvent(IntentValues.HIDE_CS_LOADING));
    }

    private static final class LazyHolder {
        static final MqttCallbackImpl INSTANCE = new MqttCallbackImpl();

        private LazyHolder() {
        }

    }

    private MqttCallbackImpl() {

    }

    public static IMqttCallback getInstance() {
        return MqttCallbackImpl.LazyHolder.INSTANCE;
    }

}
