package net.aihelp.data.localize.data;

import android.text.TextUtils;

import net.aihelp.common.Const;
import net.aihelp.common.SpKeys;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.core.util.elva.aiml.A;
import net.aihelp.data.localize.LocalizeHelper;
import net.aihelp.data.localize.util.LocalizeUtil;
import net.aihelp.data.model.faq.FaqListEntity;
import net.aihelp.data.model.faq.FaqContentEntity;
import net.aihelp.utils.FileUtil;
import net.aihelp.utils.SpUtil;
import net.aihelp.utils.Styles;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public enum FaqHelper {

    INSTANCE;

    public static final String FAQ_NOTIFICATION = "faqNotification";
    public static final String FAQ_HOT_TOPICS = "faqHotTopics";

    private static boolean isDataSourcePrepared;

    // 未加工的原始数据
    private JSONArray rawFlatFaqArray = new JSONArray();
    private Map<String, JSONArray> rawFaqMap = new HashMap<>();

    // 加工处理过的数据源
    private List<FaqListEntity> rootSections = new ArrayList<>();
    private Map<String, List<FaqListEntity>> subSectionsMap = new HashMap<>();
    private Map<String, List<FaqListEntity>> faqQuestionsMap = new HashMap<>();

    // ******************************** 对外暴露的公共方法 ********************************

    public static boolean isFaqDataAlreadyPrepared() {
        return LocalizeUtil.isAlreadyLocalized(LocalizeHelper.FLAG_FAQ_SECTION) && isDataSourcePrepared;
    }

    public JSONArray getRawFlatFaqArray() {
        return rawFlatFaqArray;
    }

    public JSONArray getRawNotification() {
        return rawFaqMap.get(FAQ_NOTIFICATION);
    }

    public JSONArray getRawHotTopics() {
        return rawFaqMap.get(FAQ_HOT_TOPICS);
    }

    public List<FaqListEntity> getRootSections() {
        return rootSections;
    }

    public List<FaqListEntity> getSubSections(String sectionId) {
        return subSectionsMap.get(sectionId);
    }

    public List<FaqListEntity> getQuestionList(String sectionId) {
        return faqQuestionsMap.get(sectionId);
    }

    /**
     * 切换语言时重置数据
     * 注意需要把全局保存的前端化地址也清空，否则就会导致文件夹错乱
     * 这样请求的时候 url 地址就不是 .json 结尾的数据，会直接去请求接口获取数据，数据不会有问题
     * <p>
     * FAQ 数据与运营模块数据不同，FAQ 有可能直接打开页面下载，所以在切换语言时必须做清空处理，否则数据就有可能乱掉
     * 比如，从中文切换到英文，但是打开 FAQ 后，显示的还是中文，此时下载动作在后台执行，需要退出再进入才能变成正确的英文
     * <p>
     * 所以如下的清空动作对于 FAQ 是必须的，但对于 OP 来说是可选的
     */
    public void reset() {
        rawFaqMap.clear();
        rawFlatFaqArray = new JSONArray();
        rootSections.clear();
        subSectionsMap.clear();
        faqQuestionsMap.clear();
        Const.FAQ_FILE = "";
        isDataSourcePrepared = false;
    }

    public synchronized void prepareDataSource() {
        prepareDataSource(null);
    }

    public synchronized void prepareDataSource(Runnable callback) {
        try {
            // 从本地读取 FAQ 文件
            String content = FileUtil.getContentFromFile(LocalizeUtil.getFileLocation(LocalizeHelper.FLAG_FAQ_SECTION));
            if (!TextUtils.isEmpty(content)) {
                JSONArray rawSectionArray = JsonHelper.getJsonArray(new JSONObject(content), "faqlist");

                // 顶级分类列表
                prepareRootSectionList(rawSectionArray);

                // 子分类列表字典：<父分类ID, List<FaqListEntity>>
                prepareSubSectionsMap(rawSectionArray);

                // FAQ问题列表字典：<FAQ所属分类ID, List<FaqListEntity>>
                prepareFaqQuestionsMap(rawSectionArray);

                // 将所有 FAQ 都放到同一个集合中，检索时使用
                flatRawFaqJsonArray(rawSectionArray);
            }

            // 数据源都准备好后，回调给调用者
            if (callback != null) {
                ApiExecutorFactory.getHandlerExecutor().runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        callback.run();
                    }
                });
            }
            isDataSourcePrepared = true;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 将 Notification 和 HotTopics 的数据都与 Section 的数据放到一起
     * 分别以 FAQ_NOTIFICATION 和 FAQ_HOT_TOPIC 做 sectionId 存储，复用 FaqHelper 的逻辑
     */
    public synchronized void prepareNotificationAndHotTopics() {
        try {
            String content = FileUtil.getContentFromFile(LocalizeUtil.getFileLocation(LocalizeHelper.FLAG_FAQ_HOT_TOPIC));
            if (!TextUtils.isEmpty(content)) {
                JSONObject jsonObject = new JSONObject(content);

                JSONArray notificationArray = JsonHelper.getJsonArray(jsonObject, "notice");
                List<FaqListEntity> notificationQuestions = new ArrayList<>();
                for (int i = 0; i < notificationArray.length(); i++) {
                    JSONObject faq = JsonHelper.getJsonObject(notificationArray, i);
                    notificationQuestions.add(getFaqListEntity(FaqListEntity.FAQ_DISPLAY_NOTIFICATION, faq));
                    faq.put("noHtmlContent", Styles.getNoTemplateFaqContent(faq.optString("content")));
                    rawFlatFaqArray.put(faq);
                }
                faqQuestionsMap.put(FAQ_NOTIFICATION, notificationQuestions);
                rawFaqMap.put(FAQ_NOTIFICATION, notificationArray);

                JSONArray faqListArray = JsonHelper.getJsonArray(jsonObject, "faqList");
                List<FaqListEntity> hotTopicQuestions = new ArrayList<>();
                for (int i = 0; i < faqListArray.length(); i++) {
                    JSONObject faq = JsonHelper.getJsonObject(faqListArray, i);
                    hotTopicQuestions.add(getFaqListEntity(FaqListEntity.FAQ_DISPLAY_HOT_TOPICS, faq));
                    faq.put("noHtmlContent", Styles.getNoTemplateFaqContent(faq.optString("content")));
                    rawFlatFaqArray.put(faq);
                }
                faqQuestionsMap.put(FAQ_HOT_TOPICS, hotTopicQuestions);
                rawFaqMap.put(FAQ_HOT_TOPICS, faqListArray);

            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public FaqContentEntity getFaqById(String faqId) {
        return getFaqById(null, faqId);
    }

    public FaqContentEntity getFaqById(String sectionId, String faqId) {
        if (TextUtils.isEmpty(sectionId)) {
            return filterFaqById(rawFlatFaqArray, faqId);
        }
        return filterFaqById(rawFaqMap.get(sectionId), faqId);
    }

    public void afterFaqEvaluated(String faqId) {
        if (TextUtils.isEmpty(faqId)) return;
        String evaluatedFaqIds = SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.EVALUATED_FAQS));
        SpUtil.getInstance().put(getLanguageBasedKey(SpKeys.EVALUATED_FAQS), String.format("%s,%s", evaluatedFaqIds, faqId));
    }

    public boolean shouldShowQuestionFooter(String faqId) {
        if (TextUtils.isEmpty(faqId)) return false;
        return !SpUtil.getInstance().getString(getLanguageBasedKey(SpKeys.EVALUATED_FAQS)).contains(faqId);
    }

    // ******************************** 对外暴露的公共方法 ********************************


    // ******************************** 内部使用的私有方法 ********************************

    /**
     * 将所有 FAQ 都放到同一个集合中，检索时使用
     */
    private void flatRawFaqJsonArray(JSONArray rawSectionArray) {
        try {
            // rawFlatFaqArray = new JSONArray();
            for (int i = 0; i < rawSectionArray.length(); i++) {
                JSONObject section = JsonHelper.getJsonObject(rawSectionArray, i);
                JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
                if (faqs.length() > 0) {
                    for (int j = 0; j < faqs.length(); j++) {
                        JSONObject faq = JsonHelper.getJsonObject(faqs, j);
                        faq.put("noHtmlContent", Styles.getNoTemplateFaqContent(faq.optString("content")));
                        rawFlatFaqArray.put(faq);
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    private boolean shouldIgnoreCurrentEntity(FaqListEntity entity) {
        if (entity == null) return true;
        for (int j = 0; j < rootSections.size(); j++) {
            FaqListEntity listEntity = rootSections.get(j);
            if (listEntity.getId().equals(entity.getId())) {
                return true;
            }
        }
        return false;
    }

    private void prepareRootSectionList(JSONArray rawSectionArray) {
        // rootSections = new ArrayList<>();
        // 遍历所有 section，将所有有子分类的条目都挑出来，然后构建一个 父分类ID >> 子分类列表 字典
        for (int i = 0; i < rawSectionArray.length(); i++) {
            JSONObject section = JsonHelper.getJsonObject(rawSectionArray, i);
            JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
            if (faqs.length() > 0) {
                String sectionBName = section.optString("sectionBName");
                FaqListEntity entity;
                if (!TextUtils.isEmpty(sectionBName)) {
                    entity = new FaqListEntity(
                            FaqListEntity.FAQ_DISPLAY_SECTION,
                            section.optString("sectionBId"),
                            section.optString("sectionBName")
                    );
                    entity.setOrder(Integer.parseInt(section.optString("sectionBOrderNo")));
                    entity.setIconUrl(section.optString("secParentImgUrl"));
                } else {
                    entity = new FaqListEntity(
                            FaqListEntity.FAQ_DISPLAY_SECTION,
                            section.optString("sectionId"),
                            section.optString("sectionName")
                    );
                    entity.setOrder(Integer.parseInt(section.optString("orderNo")));
                    entity.setIconUrl(section.optString("secImgUrl"));
                }
                if (!shouldIgnoreCurrentEntity(entity)) {
                    rootSections.add(entity);
                }
            }
        }
        Collections.sort(rootSections);
    }

    private boolean isDuplicate(List<FaqListEntity> subSections, String sectionId) {
        for (int i = 0; i < subSections.size(); i++) {
            FaqListEntity faqListEntity = subSections.get(i);
            if (faqListEntity != null && faqListEntity.getId().equals(sectionId)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 父分类ID >> 去重后的子分类列表
     */
    public void prepareSubSectionsMap(JSONArray rawSectionArray) {
        // subSectionsMap = new HashMap<>();
        for (int i = 0; i < rawSectionArray.length(); i++) {
            JSONObject section = JsonHelper.getJsonObject(rawSectionArray, i);
            JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
            if (faqs.length() > 0) {
                String parentSectionId = section.optString("sectionBId");
                String parentSectionName = section.optString("sectionBName");
                // 用 sectionBName 判断是否是父分类，但是 Map 要用 sectionBId 做 key
                if (!TextUtils.isEmpty(parentSectionName)) {
                    List<FaqListEntity> subSections = subSectionsMap.get(parentSectionId);
                    if (subSections == null) {
                        subSections = new ArrayList<>();
                    }
                    if (!isDuplicate(subSections, section.optString("sectionId"))) {
                        FaqListEntity entity = new FaqListEntity(FaqListEntity.FAQ_DISPLAY_SUB_SECTION,
                                section.optString("sectionId"), section.optString("sectionName"));
                        entity.setIconUrl(section.optString("secImgUrl"));
                        subSections.add(entity);
                        // 父分类ID >> 去重后的子分类列表
                        subSectionsMap.put(parentSectionId, subSections);
                    }
                }
            }
        }
    }

    public boolean hasSubsections(String sectionId) {
        return subSectionsMap != null && subSectionsMap.containsKey(sectionId);
    }

    /**
     * FAQ问题列表字典：<FAQ所属分类ID, List<FaqListEntity>>
     */
    private void prepareFaqQuestionsMap(JSONArray rawSectionArray) {
        // faqQuestionsMap = new HashMap<>();
        // rawFaqMap = new HashMap<>();
        for (int i = 0; i < rawSectionArray.length(); i++) {
            JSONObject section = JsonHelper.getJsonObject(rawSectionArray, i);
            JSONArray faqs = JsonHelper.getJsonArray(section, "faqs");
            if (faqs.length() > 0) {
                String sectionId = section.optString("sectionId");

                // 分类ID >> FAQ问题列表（只有 id 和标题，没有内容）
                List<FaqListEntity> faqQuestions = faqQuestionsMap.get(sectionId);
                if (faqQuestions == null) {
                    faqQuestions = new ArrayList<>();
                }
                for (int j = 0; j < faqs.length(); j++) {
                    JSONObject faq = JsonHelper.getJsonObject(faqs, j);
                    faqQuestions.add(getFaqListEntity(FaqListEntity.FAQ_DISPLAY_QUESTION_LIST, faq));
                }
                faqQuestionsMap.put(sectionId, faqQuestions);

                // 分类ID >> FAQ问题列表
                rawFaqMap.put(sectionId, faqs);
            }
        }
    }

    private FaqContentEntity filterFaqById(JSONArray faqs, String faqId) {
        if (faqs != null && faqs.length() > 0) {
            for (int j = 0; j < faqs.length(); j++) {
                JSONObject faq = JsonHelper.getJsonObject(faqs, j);
                String displayId = faq.optString("faqId");
                String mainId = faq.optString("kmMainid");
                String contentId = faq.optString("kmContentId");
                if (displayId.equals(faqId) || mainId.equals(faqId) || contentId.equals(faqId)) {
                    return getFaqContentEntity(faq);
                }
            }
        }
        return null;
    }

    private FaqListEntity getFaqListEntity(int displayType, JSONObject faq) {
        FaqListEntity entity = new FaqListEntity(displayType, faq.optString("kmMainid"), faq.optString("question"));
        entity.setIconUrl(JsonHelper.optString(faq, "imgUrl"));
        return entity;
    }

    private FaqContentEntity getFaqContentEntity(JSONObject faq) {
        FaqContentEntity faqEntity = new FaqContentEntity();
        faqEntity.setFaqMainId(faq.optString("kmMainid"));
        faqEntity.setFaqDisplayId(faq.optString("faqId"));
        faqEntity.setFaqContentId(faq.optString("kmContentId"));
        faqEntity.setFaqTitle(faq.optString("question"));
        faqEntity.setFaqKeywords(faq.optString("keyWords"));
        faqEntity.setSimilarQuestions(faq.optString("similarQuestions"));
        faqEntity.setFaqContent(faq.optString("content"));
        faqEntity.setFaqNoHtmlContent(Styles.getNoTemplateFaqContent(faq.optString("content")));
        return faqEntity;
    }

    private String getLanguageBasedKey(String spKey) {
        return String.format("%s_%s_%s", Const.APP_ID, Const.ORIGINAL_LANGUAGE, spKey);
    }

    // ******************************** 内部使用的私有方法 ********************************

}
