package net.aihelp.core.util.viewer;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Color;
import android.media.MediaPlayer;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.VideoView;

import net.aihelp.core.ui.image.Callback;
import net.aihelp.core.ui.image.Picasso;
import net.aihelp.core.ui.image.RequestCreator;
import net.aihelp.utils.DomainSupportHelper;
import net.aihelp.utils.ResResolver;
import net.aihelp.utils.TLog;

import java.io.File;
import java.util.regex.Pattern;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.appcompat.widget.AppCompatImageView;


public class ViewerLayout extends RelativeLayout {

    private View rootView;

    private String imagePath;
    private PhotoView imageView;

    private String videoPath;
    private VideoView videoView;

    private ProgressBar progressBar;

    public ViewerLayout(@NonNull Context context) {
        super(context);
        init(context);
    }

    public ViewerLayout(@NonNull Context context, @Nullable AttributeSet attrs) {
        super(context, attrs);
        init(context);
    }

    public ViewerLayout(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init(context);
    }

    private void init(Context context) {
        rootView = LayoutInflater.from(context).inflate(ResResolver.getLayoutId("aihelp_dialog_media_viewer"), this);

        setGravity(Gravity.CENTER);
        setBackgroundColor(getBackgroundColorByAlpha(255));

        imageView = rootView.findViewById(ResResolver.getViewId("aihelp_image_view"));
        videoView = rootView.findViewById(ResResolver.getViewId("aihelp_video_view"));
        videoView.setBackgroundColor(Color.BLACK);
        progressBar = rootView.findViewById(ResResolver.getViewId("aihelp_progress_bar"));

        videoView.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
            @Override
            public void onPrepared(MediaPlayer mp) {
                videoView.start();
                mp.setLooping(true);

                if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR1) {
                    new Handler(Looper.getMainLooper()).postDelayed(new Runnable() {
                        @Override
                        public void run() {
                            finishVideoBuffer();
                        }
                    }, 1000);
                }
            }
        });

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            videoView.setOnInfoListener(new MediaPlayer.OnInfoListener() {

                @Override
                public boolean onInfo(MediaPlayer mp, int what, int extra) {
                    if (what == MediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START) {
                        finishVideoBuffer();
                        return true;
                    }
                    return false;
                }

            });
        }

    }

    private void finishVideoBuffer() {
        videoView.seekTo(0);
        progressBar.setVisibility(GONE);
        imageView.setVisibility(GONE);
        videoView.setBackgroundColor(Color.TRANSPARENT);
    }

    public void show() {

        if (!TextUtils.isEmpty(imagePath)) {

            if (!TextUtils.isEmpty(videoPath)) {
                imageView.enableGesture(false);
                videoView.setVisibility(VISIBLE);
                videoView.setVideoPath(videoPath);
            } else {
                videoView.setVisibility(GONE);
            }

            Picasso.get().load(imagePath).into(imageView, new Callback() {
                @Override
                public void onSuccess() {
                    if (TextUtils.isEmpty(videoPath)) {
                        progressBar.setVisibility(GONE);
                    }
                }

                @Override
                public void onError(Exception e) {
                    if (TextUtils.isEmpty(videoPath)) {
                        progressBar.setVisibility(GONE);
                    }
                }
            });

        }
    }

    int getBackgroundColorByAlpha(float alpha) {
        int bgColor = Color.BLACK;
        return Color.argb(Math.round(alpha), Color.red(bgColor), Color.green(bgColor), Color.blue(bgColor));
    }

    public void updateImageResource(String imagePath) {
        this.imagePath = DomainSupportHelper.correctDomain(imagePath);
    }

    public void updateVideoResource(String videoPath) {
        this.videoPath = DomainSupportHelper.correctDomain(videoPath);
    }

    @SuppressLint("ClickableViewAccessibility")
    public void setOnChildViewClickedListener(final OnClickListener onClickListener) {
        if (rootView != null) {
            rootView.setOnClickListener(onClickListener);
        }

        if (imageView != null) {
            imageView.setOnClickListener(onClickListener);
        }
    }

}
