package net.aihelp.core.util.logger.controller;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.text.TextUtils;
import android.util.Log;

import net.aihelp.common.Const;
import net.aihelp.core.util.logger.AIHelpLogger;
import net.aihelp.core.util.logger.model.LogMessage;
import net.aihelp.core.util.logger.LoggerDBHelper;
import net.aihelp.core.util.logger.table.LoggerTable;

import org.json.JSONArray;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;
import java.util.regex.Pattern;

public class LoggerDBController {

    private static final String TAG = LoggerDBController.class.getSimpleName();

    private static final Object syncLock = new Object();
    private static final int MAX_ROWS = 100;
    private final LoggerDBHelper loggerDBHelper;

    public void insert(final int level, final String message, final String stacktrace, final long timeStamp) {
        SQLiteDatabase db = null;
        Cursor allRowIdCursor = null;
        synchronized (syncLock) {
            try {
                db = this.loggerDBHelper.getWritableDatabase();
                db.beginTransaction();

                try {
                    allRowIdCursor = db.rawQuery("SELECT rowid FROM LOG_MESSAGES", null);
                    if (allRowIdCursor != null && allRowIdCursor.getCount() >= MAX_ROWS) {
                        allRowIdCursor.moveToFirst();
                        int firstRowId = allRowIdCursor.getInt(0);
                        db.delete(LoggerTable.TABLE_NAME, "rowid = ?", new String[]{String.valueOf(firstRowId)});
                    }
                } catch (Exception e) {
                    Log.w(TAG, "Error in rotation of logs + " + e.getMessage());
                    db.delete(LoggerTable.TABLE_NAME, (String) null, (String[]) null);
                }

                db.insert(LoggerTable.TABLE_NAME, (String) null, toContentValues(level, message, stacktrace, timeStamp));
                db.setTransactionSuccessful();
            } catch (Exception e) {
                Log.e(TAG, "Error inserting log : ", e);
            } finally {
                try {
                    if (db != null) {
                        db.endTransaction();
                    }
                } catch (Exception e) {
                    Log.e(TAG, "Error inserting log inside finally block: ", e);
                }

                if (allRowIdCursor != null) {
                    allRowIdCursor.close();
                }
            }
        }
    }

    public JSONArray getCachedLogs() {
        Cursor cursor = null;
        synchronized (syncLock) {
            try {
                SQLiteDatabase db = this.loggerDBHelper.getReadableDatabase();
                cursor = db.rawQuery("SELECT * FROM LOG_MESSAGES", null);
                return fromCursor(cursor);
            } catch (Exception e) {
                Log.e(TAG, "Error getting all log messages : ", e);
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
            return new JSONArray();
        }
    }

    public void deleteAll() {
        synchronized (syncLock) {
            try {
                SQLiteDatabase db = this.loggerDBHelper.getWritableDatabase();
                db.execSQL("DELETE FROM LOG_MESSAGES");
            } catch (Exception var4) {
                Log.e(TAG, "Error deleting all logs from db", var4);
            }

        }
    }

    private ContentValues toContentValues(int level, String message, String stacktrace, long timeStamp) {
        ContentValues values = new ContentValues();
        try {
            if (!TextUtils.isEmpty(message) && message.length() > 5000) {
                message = message.substring(0, 5000);
            }
            if (!TextUtils.isEmpty(stacktrace) && stacktrace.length() > 5000) {
                stacktrace = stacktrace.substring(0, 5000);
            }
            values.put(LoggerTable.Columns.LEVEL, level);
            values.put(LoggerTable.Columns.TIMESTAMP, timeStamp);
            values.put(LoggerTable.Columns.MESSAGE, message);
            values.put(LoggerTable.Columns.STACKTRACE, stacktrace);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return values;
    }

    private JSONArray fromCursor(Cursor c) {
        JSONArray logArray = new JSONArray();
        try {
            if (c != null && c.moveToFirst()) {
                while (!c.isAfterLast()) {
                    JSONObject jsonObject = new JSONObject();
                    String level = getLogLevel(c.getInt(c.getColumnIndex(LoggerTable.Columns.LEVEL)));
                    String timestamp = c.getString(c.getColumnIndex(LoggerTable.Columns.TIMESTAMP));
                    String message = c.getString(c.getColumnIndex(LoggerTable.Columns.MESSAGE));
                    String stacktrace = c.getString(c.getColumnIndex(LoggerTable.Columns.STACKTRACE));
                    jsonObject.put("level", level);
                    jsonObject.put("timestamp", Pattern.matches("[1-9]*", timestamp) ?
                            Long.parseLong(timestamp) : 0);
                    jsonObject.put("message", message);
                    jsonObject.put("stacktrace", stacktrace);
                    logArray.put(jsonObject);
                    c.moveToNext();
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return logArray;
    }

    private static final class LazyHolder {
        static final LoggerDBController INSTANCE = new LoggerDBController();
    }

    public static LoggerDBController getInstance() {
        return LoggerDBController.LazyHolder.INSTANCE;
    }

    private String getLogLevel(int level) {
        switch (level) {
            case AIHelpLogger.LEVEL_FATAL:
                return "fatal";
            case AIHelpLogger.LEVEL_ERROR:
                return "error";
            case AIHelpLogger.LEVEL_WARN:
            default:
                return "warn";
        }
    }

    private LoggerDBController() {
        loggerDBHelper = LoggerDBHelper.getInstance();
    }

}
