package net.aihelp.config;

import android.text.TextUtils;

import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.utils.DeviceUuidFactory;

import org.json.JSONObject;

import java.util.Arrays;
import java.util.Iterator;

public class UserConfig {

    private String userId;
    private String userName;
    private String serverId;
    private String userTags;
    private String formatCustomData;
    private boolean isSyncCrmInfo;
    private String pushToken;
    private PushPlatform pushPlatform;

    public static class Builder {
        private String userId = "";
        private String userName = "";
        private String serverId = "";
        private String userTags = "";
        private String customData = "";
        private boolean isSyncCrmInfo;
        private String pushToken = "";
        private PushPlatform pushPlatform = PushPlatform.FIREBASE;

        /**
         * Set the identifier uniquely identifies current user.
         * <p>
         * If you did not assign values to this field, you will get a unique deviceId to identify current user.
         *
         * @param userId current uid, device id by default, can not be '0' or '-1'
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setUserId(String userId) {
            if (!TextUtils.isEmpty(userId) && !"0".equals(userId) && !"-1".equals(userId)) {
                this.userId = userId.trim()
                        .replace("/", "%2F")
                        .replace("+", "%2B")
                        .replace("#", "%23")
                        .replace(" ", "%20")
                        .replace("|", "%7C");
            }
            return this;
        }

        /**
         * The name of current user.
         *
         * @param userName current user name, 'anonymous' by default
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setUserName(String userName) {
            if (!TextUtils.isEmpty(userName)) {
                this.userName = userName;
            }
            return this;
        }

        /**
         * Set serverId of current user.
         *
         * @param serverId current user's server id, '-1' by default.
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setServerId(String serverId) {
            if (!TextUtils.isEmpty(serverId) && !"0".equals(serverId) && !"-1".equals(serverId)) {
                this.serverId = serverId;
            }
            return this;
        }

        /**
         * Set tags of current user.
         * <p>
         * If you assign this field with existing tags from aihelp admin dashboard, the tickets
         * created by current user will take these tags by default.
         * <p>
         * NOTE that the parameter can be a tag list, you should split different tags by ','.
         *
         * @param userTags tag list for current user, such as: "vip0,pay1"
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setUserTags(String userTags) {
            if (!TextUtils.isEmpty(userTags)) {
                this.userTags = userTags;
            }
            return this;
        }

        /**
         * Set custom meta data you want to see in the aihelp admin dashboard.
         * <p>
         * NOTE that this is a JSONString contains key-value maps, not just a normal string.
         *
         * @param customDataJsonString customData in jsonString, such as: {"key":"value", "key":"value"}
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setCustomData(String customDataJsonString) {
            if (!TextUtils.isEmpty(customDataJsonString)) {
                this.customData = customDataJsonString;
            }
            return this;
        }

        /**
         * If you set this to true, AIHelp will collect necessary information to display in dashboard.
         * Such as android package name, application name, server id, device model, battery status,
         * network type, carrier, country code, etc.
         *
         * @param privacyPolicyAccepted whether user has accepted all privacy policies
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setPrivacyPolicyAccepted(boolean privacyPolicyAccepted) {
            Const.PRIVACY_POLICY_ACCEPTED = privacyPolicyAccepted;
            return this;
        }

        /**
         * If you set this to true, when you update current user's information,
         * the sdk will sync user's information to you crm database.
         *
         * @param syncCrmInfo whether to sync user information to your crm database.
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setSyncCrmInfo(boolean syncCrmInfo) {
            isSyncCrmInfo = syncCrmInfo;
            return this;
        }

        /**
         * Set the exact push token of your user.
         *
         * @param pushToken The pushToken you received from the third-party push platform's notification servers.
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setPushToken(String pushToken) {
            this.pushToken = pushToken;
            return this;
        }

        /**
         * Set the exact push platform of your application.
         *
         * @param pushPlatform the specific push platform, must be of
         *                     {@link PushPlatform#APNS}, {@link PushPlatform#FIREBASE},
         *                     {@link PushPlatform#JPUSH}, {@link PushPlatform#GETUI}
         * @return This Builder object to allow for chaining of calls to set methods
         */
        public Builder setPushPlatform(PushPlatform pushPlatform) {
            this.pushPlatform = pushPlatform;
            return this;
        }

        private String getFormattedCustomData() {
            JSONObject returnData = new JSONObject();
            try {
                JSONObject jsonObject = new JSONObject();
                if (!TextUtils.isEmpty(userTags)) {
                    String[] userTagsArray = this.userTags.trim().split(",");
                    if (userTagsArray.length > 0) {
                        jsonObject.put("elva-tags", Arrays.asList(userTagsArray));
                        jsonObject.put("hs-tags", userTags);
                    }
                }

                try {
                    if (!TextUtils.isEmpty(customData)) {
                        JSONObject customJsonObject = new JSONObject(customData);
                        Iterator<String> keys = customJsonObject.keys();
                        while (keys.hasNext()) {
                            String key = keys.next();
                            jsonObject.put(key, customJsonObject.opt(key));
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }

                returnData.put("elva-custom-metadata", jsonObject);
                returnData.put("hs-custom-metadata", new JSONObject(jsonObject.toString()));

            } catch (Exception e) {
                e.printStackTrace();
            }
            return returnData.toString();
        }

        public UserConfig build(String userId, String userName, String serverId, String userTags, String customData, boolean isSyncCrmInfo) {
            return setUserId(userId).setUserName(userName).setServerId(serverId).setUserTags(userTags)
                    .setCustomData(customData).setSyncCrmInfo(isSyncCrmInfo).build();
        }

        public UserConfig build(String userId, String userName, String serverId, String userTags,
                                String customData, boolean isSyncCrmInfo,
                                String pushToken, PushPlatform pushPlatform) {
            return setUserId(userId).setUserName(userName).setServerId(serverId).setUserTags(userTags)
                    .setCustomData(customData).setSyncCrmInfo(isSyncCrmInfo)
                    .setPushToken(pushToken).setPushPlatform(pushPlatform).build();
        }

        public UserConfig build() {
            return new UserConfig(userId, userName, serverId, userTags, getFormattedCustomData(), isSyncCrmInfo, pushToken, pushPlatform);
        }

    }

    public boolean isSyncCrmInfo() {
        return isSyncCrmInfo;
    }

    public String getUserId() {
        if (TextUtils.isEmpty(userId)) {
            return UserProfile.USER_ID;
        }
        return userId;
    }

    public String getUserName() {
        if (TextUtils.isEmpty(userName)) {
            return UserProfile.USER_NAME;
        }
        return userName;
    }

    public String getServerId() {
        if (TextUtils.isEmpty(serverId)) {
            return UserProfile.SERVER_ID;
        }
        return serverId;
    }

    public String getUserTags() {
        return userTags;
    }

    public String getFormatCustomData() {
        if (TextUtils.isEmpty(formatCustomData)) {
            return UserProfile.CUSTOM_DATA;
        }
        return formatCustomData;
    }

    public String getPushToken() {
        return pushToken;
    }

    public PushPlatform getPushPlatform() {
        return pushPlatform;
    }

    private UserConfig(String userId, String userName, String serverId, String userTags, String formatCustomData, boolean isSyncCrmInfo) {
        this.userId = userId;
        this.userName = userName;
        this.serverId = serverId;
        this.userTags = userTags;
        this.formatCustomData = formatCustomData;
        this.isSyncCrmInfo = isSyncCrmInfo;
    }

    private UserConfig(String userId, String userName, String serverId, String userTags, String formatCustomData, boolean isSyncCrmInfo, String pushToken, PushPlatform pushPlatform) {
        this.userId = userId;
        this.userName = userName;
        this.serverId = serverId;
        this.userTags = userTags;
        this.formatCustomData = formatCustomData;
        this.isSyncCrmInfo = isSyncCrmInfo;
        this.pushToken = pushToken;
        this.pushPlatform = pushPlatform;
    }

    @Override
    public String toString() {
        return "UserConfig{" +
                "userId='" + userId + '\'' +
                ", userName='" + userName + '\'' +
                ", serverId='" + serverId + '\'' +
                ", userTags='" + userTags + '\'' +
                ", formatCustomData='" + formatCustomData + '\'' +
                ", isSyncCrmInfo=" + isSyncCrmInfo +
                '}';
    }
}
