package net.aihelp.utils;

import android.content.Context;
import android.text.TextUtils;

import net.aihelp.common.API;
import net.aihelp.common.Const;
import net.aihelp.config.AIHelpContext;
import net.aihelp.core.net.http.AIHelpRequest;
import net.aihelp.core.net.http.callback.ReqCallback;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.db.AIHelpDBHelper;
import net.aihelp.ui.helper.ElvaBotHelper;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.File;
import java.util.regex.Pattern;

import androidx.annotation.RestrictTo;


@RestrictTo(RestrictTo.Scope.LIBRARY)
public class LocalizeHelper {

    public static final int FLAG_FAQ = 1001;
    public static final int FLAG_OP = 1002;
    public static final int FLAG_STORY = 1003;

    public static boolean isAIMLLocalized() {
        return isAlreadyLocalized(FLAG_STORY);
    }

    public static String getFileLocation(int mode) {
        String dirPath = "";
        File filesDir = AIHelpContext.getInstance().getContext().getFilesDir();
        if (filesDir != null) {
            dirPath = filesDir.getAbsolutePath() + getFolderName(mode) + Const.TARGET_LAN;
            File dirFile = new File(dirPath);
            if (!dirFile.exists()) {
                if (dirFile.mkdirs()) {
                    return dirFile.getAbsolutePath() + File.separator + getFileName(mode);
                }
            }
        }
        return dirPath + File.separator + getFileName(mode);
    }

    public static String getUrl(int mode) {
        return API.CDN_URL + getFolderName(mode) + Const.APP_ID + File.separator + getFileName(mode);
    }

    private static String getFolderName(int mode) {
        String middle = "";
        switch (mode) {
            case FLAG_STORY:
                middle = "/AIML/";
                break;
            case FLAG_FAQ:
                middle = "/FAQ/";
                break;
            case FLAG_OP:
                middle = "/OPerMode/";
                break;
        }
        return middle;
    }

    private static String getFileName(int mode) {
        String fileName = "";
        switch (mode) {
            case FLAG_STORY:
                fileName = Const.STORY_FILE;
                break;
            case FLAG_FAQ:
                fileName = Const.FAQ_FILE;
                break;
            case FLAG_OP:
                fileName = Const.OP_FILE;
                break;
        }
        return fileName;
    }

    // 判断当前本地有没有初始化返回的那几个文件
    public static boolean isAlreadyLocalized(int mode) {
        try {
            String path = LocalizeHelper.getFileLocation(mode);
            File file = new File(path);
            if (file.exists()) {
                return true;
            } else {
                File parentFile = file.getParentFile();
                if (parentFile != null) {
                    File[] files = parentFile.listFiles();
                    if (files != null && files.length > 0) {
                        for (File oldFile : files) {
                            if (oldFile.delete()) {
                                return false;
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            TLog.e("LocalizeHelper check localize status failed.");
        }
        return false;
    }

    public static void goFetchLocalizeData(Context context) {
        goFetchElvaBotData(context);
        if (Const.TOGGLE_LOCALIZE_VIA_INIT) {
            getLocalizeDataFromUrl(FLAG_FAQ);
        }
        getLocalizeDataFromUrl(FLAG_OP);
    }

    public static void goFetchElvaBotData(final Context context) {
        AIHelpRequest.getInstance().requestDownloadFile(LocalizeHelper.FLAG_STORY, new ReqCallback<String>() {
            @Override
            public void onAsyncReqSuccess(String result) {
                ElvaBotHelper.initBot(context);
            }
        });
    }

    public static void getLocalizeDataFromUrl(final int mode) {
        if (mode == FLAG_FAQ && AIHelpDBHelper.getInstance().isFaqStoredSuccessfully()) return;
        if (mode == FLAG_OP && AIHelpDBHelper.getInstance().isOperateStoredSuccessfully()) return;
        String targetLocalizeUrl = getUrl(mode); // This url should always be .json file
        if (Pattern.compile(".+\\.(json)$").matcher(targetLocalizeUrl).matches()) {
            AIHelpRequest.getInstance().requestGetByAsync(targetLocalizeUrl, null, new ReqCallback<String>() {
                @Override
                public void onAsyncReqSuccess(String result) {
                    try {
                        if (TextUtils.isEmpty(result)) {
                            getDataAfterLocalizeFailed(mode);
                        } else {
                            JSONArray jsonArray = JsonHelper.getJsonArray(new JSONObject(result), "faqlist");
                            if (mode == FLAG_FAQ) {
                                AIHelpDBHelper.getInstance().storeFaqList(jsonArray);
                            } else {
                                AIHelpDBHelper.getInstance().storeOperateFaqs(jsonArray);
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public boolean onFailure(int errorCode, String errorMsg) {
                    getDataAfterLocalizeFailed(mode);
                    return false;
                }
            });
        }
    }

    private static void getDataAfterLocalizeFailed(final int mode) {
        String url = mode == FLAG_FAQ ? API.FAQ_URL : API.OP_URL;
        AIHelpRequest.getInstance().requestGetByAsync(url, null, new ReqCallback<JSONObject>() {
            @Override
            public void onAsyncReqSuccess(JSONObject jsonObject) {
                JSONArray faqList = JsonHelper.getJsonArray(jsonObject, "faqlist");
                if (mode == FLAG_FAQ) {
                    AIHelpDBHelper.getInstance().storeFaqList(faqList);
                } else {
                    AIHelpDBHelper.getInstance().storeOperateFaqs(faqList);
                }
            }
        });
    }

}
