package net.aihelp.utils;

import android.content.Context;
import android.content.SharedPreferences;
import android.provider.Settings;
import android.telephony.TelephonyManager;
import android.text.TextUtils;

import net.aihelp.config.AIHelpContext;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UnsupportedEncodingException;
import java.util.Random;
import java.util.UUID;

import androidx.annotation.RestrictTo;

/**
 * Get unique device id for different versions.
 * <p>
 * 1、In version 1.x，deviceId is ssuid in sharedPreference;
 * 2、In versions before 2.1.10, dau deviceId is in sp, while other deviceIds is in external storage;
 * 3、In versions after 2.1.10, deviceId is INSTALLATION in file folder.
 */
@RestrictTo(RestrictTo.Scope.LIBRARY)
public class DeviceUuidFactory {

    private static String sID = null;

    private static final String INSTALLATION = "INSTALLATION";

    public synchronized static String id(Context context) {
        if (context == null) {
            context = AIHelpContext.getInstance().getContext();
            if (context == null) {
                sID = UUID.randomUUID().toString();
                return sID;
            }
        }

        if (sID == null) {

            // deviceId in v1.x
            String deviceIdOf1x = SpUtil.getInstance().get1_xDeviceId();

            // deviceId in v2.x
            File installation = new File(context.getFilesDir(), INSTALLATION);

            try {
                // If v2.x's deviceId exists, meaning that the app has already upgraded to v2.x.
                // We should read the deviceId to make everything goes well.
                if (installation.exists()) {
                    sID = readInstallationFile(installation);
                } else
                    // If it's not exist, we should check whether there is v1.x's deviceId.
                    // If v1.x's deviceId is non-null, then it should be upgrading from v1.x.
                    // We should just use the original deviceId and every body will be happy.
                    if (!TextUtils.isEmpty(deviceIdOf1x)) {
                        sID = deviceIdOf1x;
                    } else {
                        // If v1.x's deviceId can't be found,
                        // we should create a deviceId in internal storage.
                        writeInstallationFile(installation);
                        sID = readInstallationFile(installation);
                    }
            } catch (Exception e) {
                sID = "" + UUID.randomUUID().toString();
                return sID;
            }
        }
        return sID;
    }

    private static String readInstallationFile(File installation) throws IOException {
        RandomAccessFile f = new RandomAccessFile(installation, "r");
        byte[] bytes = new byte[(int) f.length()];
        f.readFully(bytes);
        f.close();
        return new String(bytes);

    }

    private static void writeInstallationFile(File installation) throws IOException {
        FileOutputStream out = new FileOutputStream(installation);
        String id = UUID.randomUUID().toString();
        out.write(id.getBytes());
        out.close();
    }

}
