package net.aihelp.utils;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.content.res.Resources;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.Build;
import android.os.Environment;
import android.os.StatFs;
import android.telephony.TelephonyManager;
import android.text.TextUtils;
import android.text.format.DateFormat;
import android.util.DisplayMetrics;

import net.aihelp.common.Const;
import net.aihelp.common.UserProfile;
import net.aihelp.config.AIHelpContext;
import net.aihelp.config.enums.PushPlatform;
import net.aihelp.core.net.json.JsonHelper;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.data.model.PrivacyControlEntity;

import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.TimeZone;

import androidx.annotation.RestrictTo;

@RestrictTo(RestrictTo.Scope.LIBRARY)
public class DeviceInfoUtil {

    private final Context context;

    private DeviceInfoUtil(Context context) {
        this.context = context;
    }

    private static DeviceInfoUtil sInstance;

    public static DeviceInfoUtil getInstance() {
        if (sInstance == null) {
            sInstance = new DeviceInfoUtil(AIHelpContext.getInstance().getContext());
        }
        return sInstance;
    }

    public String getSimCountryIso() {
        TelephonyManager tm = (TelephonyManager) this.context.getSystemService(Context.TELEPHONY_SERVICE);
        return tm == null ? "" : tm.getSimCountryIso();
    }

    public String getTimeStamp() {
        return (new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'", Locale.ENGLISH)).format(new Date());
    }

    public String getCarrierName() {
        TelephonyManager tm = (TelephonyManager) this.context.getSystemService(Context.TELEPHONY_SERVICE);
        return tm == null ? "" : tm.getNetworkOperatorName();
    }

    public String getNetworkType() {
        String type = "Unknown";
        try {
            ConnectivityManager cm = (ConnectivityManager) this.context.getSystemService(Context.CONNECTIVITY_SERVICE);
            if (cm != null) {
                NetworkInfo ani = cm.getActiveNetworkInfo();
                if (ani != null) {
                    type = ani.getTypeName();
                }
            }
        } catch (SecurityException var4) {
            type = null;
        }
        return type;
    }

    public String getBatteryStatus() {
        IntentFilter filter = new IntentFilter("android.intent.action.BATTERY_CHANGED");
        Intent batteryStatus = this.context.registerReceiver(null, filter);
        if (batteryStatus == null) {
            return "Not charging";
        } else {
            int status = batteryStatus.getIntExtra("status", -1);
            boolean isCharging = status == 2 || status == 5;
            return isCharging ? "Charging" : "Not charging";
        }
    }

    public String getBatteryLevel() {
        IntentFilter filter = new IntentFilter("android.intent.action.BATTERY_CHANGED");
        Intent batteryStatus = this.context.registerReceiver(null, filter);
        if (batteryStatus == null) {
            return "";
        } else {
            int level = batteryStatus.getIntExtra("level", -1);
            int scale = batteryStatus.getIntExtra("scale", -1);
            int batteryPct = (int) ((float) level / (float) scale * 100.0F);
            return batteryPct + "%";
        }
    }

    public String getTotalDiskSpace() {
        double bytesInOneGB = 1073741824.0D;
        StatFs phoneStat = new StatFs(Environment.getDataDirectory().getPath());
        double total_phone_memory;
        if (Build.VERSION.SDK_INT >= 18) {
            total_phone_memory = (double) phoneStat.getBlockCountLong() * (double) phoneStat.getBlockSizeLong() / bytesInOneGB;
            total_phone_memory = (double) Math.round(total_phone_memory * 100.0D) / 100.0D;
        } else {
            total_phone_memory = (double) phoneStat.getBlockCount() * (double) phoneStat.getBlockSize() / bytesInOneGB;
            total_phone_memory = (double) Math.round(total_phone_memory * 100.0D) / 100.0D;
        }
        return total_phone_memory + "GB";
    }

    public String getRemainDiskSpace() {
        double bytesInOneGB = 1073741824.0D;
        StatFs phoneStat = new StatFs(Environment.getDataDirectory().getPath());
        double free_phone_memory;
        if (Build.VERSION.SDK_INT >= 18) {
            free_phone_memory = (double) phoneStat.getAvailableBlocksLong() * (double) phoneStat.getBlockSizeLong() / bytesInOneGB;
            free_phone_memory = (double) Math.round(free_phone_memory * 100.0D) / 100.0D;
        } else {
            free_phone_memory = (double) phoneStat.getAvailableBlocks() * (double) phoneStat.getBlockSize() / bytesInOneGB;
            free_phone_memory = (double) Math.round(free_phone_memory * 100.0D) / 100.0D;
        }
        return free_phone_memory + "GB";
    }

    private static JSONObject getHostGameInfo(PrivacyControlEntity entity) {
        String hostPackageName = "unknown";
        String hostAppVersion = "0.0.0";
        String hostApplicationName = "unknown";
        try {
            Context context = AIHelpContext.getInstance().getContext();
            hostPackageName = context.getPackageName();
            ApplicationInfo ai = context.getPackageManager().getApplicationInfo(hostPackageName, 0);
            hostAppVersion = context.getPackageManager().getPackageInfo(hostPackageName, 0).versionName;
            hostApplicationName = (String) context.getPackageManager().getApplicationLabel(ai);
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
        JSONObject params = new JSONObject();
        try {
            if (entity != null && entity.getApplicationIdentifier()) {
                params.put("Application_Identifier", hostPackageName);
            }
            if (entity != null && entity.getApplicationVersion()) {
                params.put("Application_Version", hostAppVersion);
            }
            if (entity != null && entity.getApplicationName()) {
                params.put("Name", hostApplicationName);
            }
            if (entity != null && entity.getServerId()) {
                params.put("ServerId", UserProfile.SERVER_ID);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return params;
    }

    private JSONObject getHardwareInfo(PrivacyControlEntity entity) {
        JSONObject hardwareInfo = new JSONObject();
        try {
            hardwareInfo.put("DeviceId", DeviceUuidFactory.id(context));
            if (entity != null && entity.getDeviceModel()) {
                hardwareInfo.put("Device_Model", Build.MODEL);
            }
            if (entity != null && entity.getBatteryPower()) {
                hardwareInfo.put("Battery_Level", getBatteryLevel());
            }
            if (entity != null && entity.getBatteryStatus()) {
                hardwareInfo.put("Battery_Status", getBatteryStatus());
            }
            if (entity != null && entity.getFreeSpacePhone()) {
                hardwareInfo.put("Free_Space_Phone", getRemainDiskSpace());
            }
            if (entity != null && entity.getTotalSpacePhone()) {
                hardwareInfo.put("Total_Space_Phone", getTotalDiskSpace());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return hardwareInfo;
    }

    private JSONObject getOtherInfo(PrivacyControlEntity entity) {
        JSONObject otherInfo = new JSONObject();
        try {
            otherInfo.put("Platform", "android");
            otherInfo.put("Language", Const.TARGET_LAN);
            otherInfo.put("SDK_Version", Const.SDK_VERSION);
            if (entity != null && entity.getOsVersion()) {
                otherInfo.put("OS_Version", Build.VERSION.RELEASE);
            }
            if (entity != null && entity.getNetworkType()) {
                otherInfo.put("Network_Type", getNetworkType());
            }
            if (entity != null && entity.getOperator()) {
                otherInfo.put("Carrier", getCarrierName());
            }
            if (entity != null && entity.getCountryCode()) {
                otherInfo.put("Country_Code", getSimCountryIso());
            }
            JSONObject pushDeviceTokenObj = new JSONObject(getPushDeviceToken());
            Iterator<String> keys = pushDeviceTokenObj.keys();
            while (keys.hasNext()) {
                String key = keys.next();
                otherInfo.put(key, pushDeviceTokenObj.opt(key));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return otherInfo;
    }

    private HashMap<String, String> getPushDeviceToken() {
        HashMap<String, String> pushTokens = new HashMap<>();
        if (!TextUtils.isEmpty(Const.PUSH_INFO) && Const.PUSH_INFO.contains("|")) {
            String pushToken = Const.PUSH_INFO.split("\\|")[0];
            int pushPlatform = Integer.parseInt(Const.PUSH_INFO.split("\\|")[1]);
            if (pushPlatform == PushPlatform.APNS.getValue()) {
                pushTokens.put("ApnsDeviceToken", pushToken);
            }

            if (pushPlatform == PushPlatform.FIREBASE.getValue()) {
                pushTokens.put("DeviceToken", pushToken);
            }

            if (pushPlatform == PushPlatform.JPUSH.getValue()) {
                pushTokens.put("JDeviceToken", pushToken);
            }

            if (pushPlatform == PushPlatform.GETUI.getValue()) {
                pushTokens.put("GeTuiDeviceToken", pushToken);
            }
        }
        return pushTokens;
    }

    private String fillStoryNodeToCustomData() {

        if (TextUtils.isEmpty(Const.CUSTOM_STORY_NODE) || MqttConfig.getInstance().getLoginType() == MqttConfig.TYPE_CONVERSATION) {
            return UserProfile.CUSTOM_DATA;
        }

        try {
            JSONObject customData = new JSONObject(UserProfile.CUSTOM_DATA);
            JSONObject elvaMetaData = JsonHelper.getJsonObject(customData, "elva-custom-metadata");
            elvaMetaData.put("anotherWelcomeText", Const.CUSTOM_STORY_NODE);

            JSONObject returnData = new JSONObject();
            returnData.put("elva-custom-metadata", elvaMetaData);
            returnData.put("hs-custom-metadata", new JSONObject(elvaMetaData.toString()));
            return returnData.toString();
        } catch (Exception e) {
            e.printStackTrace();
            return UserProfile.CUSTOM_DATA;
        }
    }

    public JSONObject getGameInfo() {
        JSONObject gameInfo = new JSONObject();
        try {
            PrivacyControlEntity entity;
            if (!TextUtils.isEmpty(Const.PRIVACY_CONTROL_DATA)) {
                entity = JsonHelper.toJavaObject(Const.PRIVACY_CONTROL_DATA,
                        PrivacyControlEntity.class);
            } else {
                entity = new PrivacyControlEntity();
            }
            gameInfo.put("APPLICATION", getHostGameInfo(entity));
            gameInfo.put("HARDWARE", getHardwareInfo(entity));
            gameInfo.put("OTHER", getOtherInfo(entity));
            gameInfo.put("CUSTOMDATA", fillStoryNodeToCustomData());
        } catch (Exception e) {
            e.printStackTrace();
        }
        return gameInfo;
    }


}

