package net.aihelp.utils;

import android.content.res.Resources;

import net.aihelp.R;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;

public class DateFormatUtil {

    /**
     *  相等就是同一天，传入时间比该时间大就说明在区间内
     *
     *  假设 nDays 传入 1，Calendar -1 即昨天，那么 veryStartTime 是昨天最早的那一刻；
     *  传入的时间如果是昨天，c.getTimeInMillis() 是昨天最早的一刻，满足 == 条件；
     *  传入的时间如果是今天，c.getTimeInMillis() 返回的就是今天最早的一刻，满足 < 条件；
     *
     *  假设 nDays 传入 6，Calendar -6 就是 6 天前，即今天是第七天，那么 veryStartTime 是往回数 6 天，即上周日最早的那一刻；
     *  今天是周六
     *  传入的时间如果是昨天，c.getTimeInMillis() 是昨天最早的一刻，满足 < 条件；
     *  传入的时间如果是上周日，c.getTimeInMillis() 返回的就是上周日最早的一刻，满足 == 条件；
     *  传入的时间如果是上周六，c.getTimeInMillis() 返回的就是上周六最早的一刻，不满足条件；
     *
     * @param nDays 往回数几天
     * @param timestamp 当前传入的时间戳
     * @return [几天前那天的起始时间, 传入时间当天的起始时间]
     */
    private static long[] goBackNDays(int nDays, long timestamp) {

        long[] results = new long[2];
        Calendar c = Calendar.getInstance();

        // 几天前那天的最早时间
        clearCalendar(c, Calendar.HOUR_OF_DAY, Calendar.MINUTE, Calendar.SECOND, Calendar.MILLISECOND);
        c.add(Calendar.DAY_OF_MONTH, -nDays);
        results[0] = c.getTimeInMillis();

        // 指定时间戳当天最早的时间
        c.setTimeInMillis(timestamp);
        clearCalendar(c, Calendar.HOUR_OF_DAY, Calendar.MINUTE, Calendar.SECOND, Calendar.MILLISECOND);
        results[1] = c.getTimeInMillis();

        return results;
    }

    private static void clearCalendar(Calendar c, int... fields) {
        for (int f : fields) {
            c.set(f, 0);
        }
    }

    public static boolean isToday(long timeStamp) {
        long[] time = goBackNDays(0, timeStamp);
        return time[0] == time[1];
    }

    public static boolean isYesterday(long timeStamp) {
        long[] time = goBackNDays(1, timeStamp);
        return time[0] == time[1];
    }

    public static boolean isBackNDays(int n, long timeStamp) {
        long[] time = goBackNDays(n, timeStamp);
        return time[0] <= time[1];
    }

    public static String getWeekOfDate(Resources resources, long timeStamp) {

        String[] weekDays = {
                resources.getString(R.string.aihelp_sunday),
                resources.getString(R.string.aihelp_monday),
                resources.getString(R.string.aihelp_tuesday),
                resources.getString(R.string.aihelp_wednesday),
                resources.getString(R.string.aihelp_thursday),
                resources.getString(R.string.aihelp_friday),
                resources.getString(R.string.aihelp_saturday)
        };
        Calendar cal = Calendar.getInstance();
        cal.setTime(new Date(timeStamp));

        int w = cal.get(Calendar.DAY_OF_WEEK) - 1;
        if (w < 0)
            w = 0;

        return weekDays[w];
    }

    public static String dateFormat(Date date) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        String dateString = formatter.format(date);
        return dateString;
    }

    /**
     * 将"2015-08-31 21:08:06"型字符串转化为Date
     *
     * @param str
     * @return
     * @throws ParseException
     */
    public static Date StringToDate(String str) {
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        Date date = null;
        try {
            date = formatter.parse(str);
        } catch (ParseException e) {
            e.printStackTrace();
        }
        return date;
    }

    public static String CSTFormat(String str) throws ParseException {
        SimpleDateFormat formatter = new SimpleDateFormat("EEE MMM dd HH:mm:ss zzz yyyy", Locale.US);
        Date date = formatter.parse(str);
        return dateFormat(date);
    }

    public static Date LongToDare(long str) {
        return new Date(str);
    }

    public static String dateFormat(long timeStamp) {
        return dateFormat(LongToDare(timeStamp));
    }

    public static String getTodayTime(long timeStamp) {
        SimpleDateFormat formatter = new SimpleDateFormat("HH:mm");
        return formatter.format(new Date(timeStamp));
    }

    public static String getProperDate(Resources resources, long timeStamp) {
        if (isToday(timeStamp)) {
            return getTodayTime(timeStamp);
        } else if (isYesterday(timeStamp)) {
            return resources.getString(R.string.aihelp_yesterday) + " " + getTodayTime(timeStamp);
        } else if ((isBackNDays(6, timeStamp))) {
            return getWeekOfDate(resources, timeStamp) + " " + getTodayTime(timeStamp);
        } else {
            return dateFormat(timeStamp) + " " + getTodayTime(timeStamp);
        }
    }

}
