package net.aihelp.ui.widget;

import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Rect;
import android.os.Build;
import android.os.Bundle;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.FrameLayout;

import net.aihelp.R;
import net.aihelp.ui.widget.material.BottomSheetBehavior;
import net.aihelp.ui.widget.material.CoordinatorLayout;

import androidx.annotation.LayoutRes;
import androidx.annotation.NonNull;
import androidx.annotation.StyleRes;
import androidx.appcompat.app.AppCompatDialog;
import androidx.core.view.AccessibilityDelegateCompat;
import androidx.core.view.ViewCompat;
import androidx.core.view.accessibility.AccessibilityNodeInfoCompat;

public class AIHelpBottomSheetDialog extends AppCompatDialog {

    private BottomSheetBehavior<FrameLayout> mBehavior;

    private Context context;
    boolean mCancelable = true;
    private boolean mCanceledOnTouchOutside = true;
    private boolean mCanceledOnTouchOutsideSet;

    private CoordinatorLayout coordinator;
    private FrameLayout bottomSheet;
    private View mContentView;

    private Rect r = new Rect();

    public View getContentView() {
        return mContentView;
    }

    public AIHelpBottomSheetDialog(@NonNull Context context, @LayoutRes int layoutResId, boolean cancelable) {
        this(context);
        setCancelable(cancelable);
        setCanceledOnTouchOutside(cancelable);
        setContentView(layoutResId);
    }

    public AIHelpBottomSheetDialog(@NonNull Context context, @LayoutRes int layoutResId) {
        this(context);
        setContentView(layoutResId);
    }

    public AIHelpBottomSheetDialog(@NonNull Context context) {
        super(context, R.style.aihelp_theme_bottom_sheet_light_dialog);
        this.context = context;
        // We hide the title bar for any style configuration. Otherwise, there will be a gap
        // above the bottom sheet when it is expanded.
        supportRequestWindowFeature(Window.FEATURE_NO_TITLE);
    }

    protected AIHelpBottomSheetDialog(@NonNull Context context, boolean cancelable,
                                      OnCancelListener cancelListener) {
        super(context, cancelable, cancelListener);
        supportRequestWindowFeature(Window.FEATURE_NO_TITLE);
        mCancelable = cancelable;
    }

    @Override
    public void setContentView(@LayoutRes int layoutResId) {
        mContentView = wrapInBottomSheet(layoutResId, null, null);
        super.setContentView(mContentView);
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        Window window = getWindow();
        if (window != null) {
            if (Build.VERSION.SDK_INT >= 21) {
                window.clearFlags(WindowManager.LayoutParams.FLAG_TRANSLUCENT_STATUS);
                window.addFlags(WindowManager.LayoutParams.FLAG_DRAWS_SYSTEM_BAR_BACKGROUNDS);
            }
            window.setLayout(ViewGroup.LayoutParams.MATCH_PARENT,
                    ViewGroup.LayoutParams.MATCH_PARENT);
        }
    }

    @Override
    public void setContentView(View view) {
        super.setContentView(wrapInBottomSheet(0, view, null));
    }

    @Override
    public void setContentView(View view, ViewGroup.LayoutParams params) {
        super.setContentView(wrapInBottomSheet(0, view, params));
    }

    @Override
    public void setCancelable(boolean cancelable) {
        super.setCancelable(cancelable);
        if (mCancelable != cancelable) {
            mCancelable = cancelable;
            if (mBehavior != null) {
                mBehavior.setHideable(cancelable);
            }
        }
    }

    @Override
    protected void onStart() {
        super.onStart();
        if (mBehavior != null) {
            mBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
        }
    }

    @Override
    public void setCanceledOnTouchOutside(boolean cancel) {
        super.setCanceledOnTouchOutside(cancel);
        if (cancel && !mCancelable) {
            mCancelable = true;
        }
        mCanceledOnTouchOutside = cancel;
        mCanceledOnTouchOutsideSet = true;
    }

    /**
     * 添加 top 距离顶部多少的时候触发收缩效果
     *
     * @param targetLimitH int 高度限制
     */
    @SuppressWarnings("all")
    public void addSpringBackDisLimit(final int targetLimitH) {
        if (coordinator == null)
            return;
        final int totalHeight = getContext().getResources().getDisplayMetrics().heightPixels;
        final int currentH = (int) ((float) totalHeight * 0.618);
        final int leftH = totalHeight - currentH;
        coordinator.setOnTouchListener(
                new View.OnTouchListener() {
                    @Override
                    public boolean onTouch(View v, MotionEvent event) {
                        switch (event.getAction()) {
                            case MotionEvent.ACTION_MOVE:
                                // 计算相对于屏幕的 坐标
                                bottomSheet.getGlobalVisibleRect(r);
                                break;
                            case MotionEvent.ACTION_UP:
                                int limitH;
                                if (targetLimitH < 0)
                                    limitH = (leftH + currentH / 3);
                                else
                                    limitH = targetLimitH;
                                if (r.top <= limitH)
                                    if (mBehavior != null)
                                        mBehavior.setState(BottomSheetBehavior.STATE_COLLAPSED);
                                break;
                        }
                        return false;
                    }
                }
        );
    }

    private View wrapInBottomSheet(int layoutResId, View view, ViewGroup.LayoutParams params) {
        final FrameLayout container = (FrameLayout) View.inflate(getContext(),
                R.layout.aihelp_design_bottom_sheet_dialog, null);
        coordinator = container.findViewById(R.id.aihelp_coordinator);
        if (layoutResId != 0 && view == null) {
            view = getLayoutInflater().inflate(layoutResId, coordinator, false);
        }
        bottomSheet = coordinator.findViewById(R.id.design_bottom_sheet);
        bottomSheet.setBackgroundResource(R.drawable.aihelp_bg_bottom_sheet);
        mBehavior = BottomSheetBehavior.from(bottomSheet);
        mBehavior.setBottomSheetCallback(mBottomSheetCallback);
        mBehavior.setHideable(mCancelable);
        mBehavior.setPeekHeight((int) ((float) (context.getResources().getDisplayMetrics().heightPixels * 0.9)));
        if (params == null) {
            bottomSheet.addView(view);
        } else {
            bottomSheet.addView(view, params);
        }
        // We treat the CoordinatorLayout as outside the dialog though it is technically inside
        coordinator.findViewById(R.id.touch_outside).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if (mCancelable && isShowing() && shouldWindowCloseOnTouchOutside()) {
                    cancel();
                }
            }
        });
        // Handle accessibility events
        ViewCompat.setAccessibilityDelegate(bottomSheet, new AccessibilityDelegateCompat() {
            @Override
            public void onInitializeAccessibilityNodeInfo(View host,
                                                          AccessibilityNodeInfoCompat info) {
                super.onInitializeAccessibilityNodeInfo(host, info);
                if (mCancelable) {
                    info.addAction(AccessibilityNodeInfoCompat.ACTION_DISMISS);
                    info.setDismissable(true);
                } else {
                    info.setDismissable(false);
                }
            }

            @Override
            public boolean performAccessibilityAction(View host, int action, Bundle args) {
                if (action == AccessibilityNodeInfoCompat.ACTION_DISMISS && mCancelable) {
                    cancel();
                    return true;
                }
                return super.performAccessibilityAction(host, action, args);
            }
        });
        bottomSheet.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View view, MotionEvent event) {
                // coordinator intecept, 这没用
                return true;
            }
        });
        return container;
    }

    private boolean shouldWindowCloseOnTouchOutside() {
        if (!mCanceledOnTouchOutsideSet) {
            TypedArray a = getContext().obtainStyledAttributes(
                    new int[]{android.R.attr.windowCloseOnTouchOutside});
            mCanceledOnTouchOutside = a.getBoolean(0, true);
            a.recycle();
            mCanceledOnTouchOutsideSet = true;
        }
        return mCanceledOnTouchOutside;
    }

    private BottomSheetBehavior.BottomSheetCallback mBottomSheetCallback
            = new BottomSheetBehavior.BottomSheetCallback() {
        @Override
        public void onStateChanged(@NonNull View bottomSheet,
                                   @BottomSheetBehavior.State int newState) {
            if (newState == BottomSheetBehavior.STATE_HIDDEN) {
                cancel();
            }
        }

        @Override
        public void onSlide(@NonNull View bottomSheet, float slideOffset) {
        }
    };

}
