package net.aihelp.ui.webkit;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.graphics.Bitmap;
import android.net.Uri;
import android.net.http.SslError;
import android.os.Build;
import android.text.TextUtils;
import android.util.Log;
import android.webkit.SslErrorHandler;
import android.webkit.WebResourceRequest;
import android.webkit.WebResourceResponse;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.Toast;

import net.aihelp.utils.AppInfoUtil;
import net.aihelp.utils.FileUtil;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;

public class AIHelpWebViewClient extends WebViewClient {

    public static final String TAG = AIHelpWebViewClient.class.getSimpleName();
    private Context context;
    private AIHelpWebProgress webProgress;
    private ShouldOverrideUrlLoadingListener mUrlLoadingListener;

    public void setUrlLoadingListener(ShouldOverrideUrlLoadingListener mUrlLoadingListener) {
        this.mUrlLoadingListener = mUrlLoadingListener;
    }

    public AIHelpWebViewClient(Context context, AIHelpWebProgress webProgress) {
        this.context = context;
        this.webProgress = webProgress;
    }

    public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest request) {
        boolean ret = Build.VERSION.SDK_INT >= 24 && this.handleUrlClick(view, request.getUrl().toString()) || super.shouldOverrideUrlLoading(view, request);
        if (mUrlLoadingListener != null) {
            mUrlLoadingListener.handleUrlClick(ret);
        }
        return ret;
    }

    public boolean shouldOverrideUrlLoading(WebView view, String url) {
        boolean ret = Build.VERSION.SDK_INT < 24 && this.handleUrlClick(view, url) || super.shouldOverrideUrlLoading(view, url);
        if (mUrlLoadingListener != null) {
            mUrlLoadingListener.handleUrlClick(ret);
        }
        return ret;
    }

    @Override
    public void onPageStarted(WebView view, String url, Bitmap favicon) {
        super.onPageStarted(view, url, favicon);
        webProgress.show();
    }

    @Override
    public void onPageFinished(WebView view, String url) {
        super.onPageFinished(view, url);
        webProgress.hide();
    }

    // 这个方法只会响应点击，第一次加载不走这里
    private boolean handleUrlClick(WebView view, String url) {
        Context context = view.getContext();
        if (AppInfoUtil.isNetworkAvailable(context)) {
            if (!TextUtils.isEmpty(url)) {
                url = url.trim();
                Intent intent = new Intent(Intent.ACTION_VIEW);
                Uri data = Uri.parse(url);
                intent.setData(data);
                if (intent.resolveActivity(context.getPackageManager()) != null) {
                    view.loadUrl(url);
                    return true;
                }
            }

            if (url.startsWith("intent://") || url.startsWith("jsbridge://")) {
                view.loadUrl(null);
                try {
                    Intent intent = Intent.parseUri(url, Intent.URI_INTENT_SCHEME);
                    if (intent != null) {
                        intent.addCategory(Intent.CATEGORY_BROWSABLE);
                        intent.setComponent(null);
                        intent.setSelector(null);
                        PackageManager packageManager = context.getPackageManager();
                        ResolveInfo info = packageManager.resolveActivity(intent, PackageManager.MATCH_DEFAULT_ONLY);
                        if (info != null) {
                            context.startActivity(intent);
                        } else {
                            String fallbackUrl = intent.getStringExtra("browser_fallback_url");
                            view.loadUrl(fallbackUrl);
                        }
                        return true;
                    }
                } catch (URISyntaxException e) {
                    e.printStackTrace();
                }
            }

            // 加上这一行就无法加载腾讯文档了
            // view.loadUrl(url);
        } else {
            Toast.makeText(context, context.getString(net.aihelp.R.string.aihelp_network_no_connect), Toast.LENGTH_SHORT).show();
            return true;
        }
        return false;
    }

    private WebResourceResponse interceptRequest(String inputUrl) {
        URL url = null;
        File storagePath = this.context.getExternalCacheDir();

        try {
            url = new URL(inputUrl);
        } catch (Exception e) {
            e.printStackTrace();
        }

        if (url != null) {
            File saveFile = new File(storagePath, inputUrl.replace("/", "_"));
            if (saveFile.exists()) {
                try {
                    return new WebResourceResponse("", "", new FileInputStream(saveFile));
                } catch (FileNotFoundException e) {
                    e.printStackTrace();
                }
            } else if (FileUtil.isSupportedMimeType(FileUtil.getMimeType(url))) {
                FileUtil.saveFile(url, saveFile);
            }
        }

        return null;
    }

    public WebResourceResponse shouldInterceptRequest(WebView view, WebResourceRequest request) {
        if (Build.VERSION.SDK_INT >= 21) {
            WebResourceResponse response = this.interceptRequest(request.getUrl().toString());
            if (response != null) {
                return response;
            }
        }

        return super.shouldInterceptRequest(view, request);
    }

    public WebResourceResponse shouldInterceptRequest(WebView view, String inputUrl) {
        if (Build.VERSION.SDK_INT < 21) {
            WebResourceResponse response = this.interceptRequest(inputUrl);
            if (response != null) {
                return response;
            }
        }

        return super.shouldInterceptRequest(view, inputUrl);
    }

    public interface ShouldOverrideUrlLoadingListener {
        void handleUrlClick(boolean result);
    }

    @Override
    public void onReceivedSslError(WebView view, SslErrorHandler handler, SslError error) {
        super.onReceivedSslError(view, handler, error);
        String message;
        switch (error.getPrimaryError()) {
            case SslError.SSL_DATE_INVALID:
                message = "SSL_DATE_INVALID";
                break;
            case SslError.SSL_EXPIRED:
                message = "SSL_EXPIRED";
                break;
            case SslError.SSL_IDMISMATCH:
                message = "SSL_IDMISMATCH";
                break;
            case SslError.SSL_INVALID:
                message = "SSL_INVALID";
                break;
            case SslError.SSL_NOTYETVALID:
                message = "SSL_NOTYETVALID";
                break;
            case SslError.SSL_UNTRUSTED:
                message = "SSL_UNTRUSTED";
                break;
            default:
                message = "SslError unknown";
        }
        Log.d(TAG, "onReceivedSslError: " + message);
    }
}
