package net.aihelp.ui.helper;

import android.text.TextUtils;

import net.aihelp.common.Const;
import net.aihelp.data.model.ChatHistoryEntity;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.data.model.cs.ElvaBotMsg;
import net.aihelp.utils.TLog;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;


public class ConversationHelper {

    private static int sFakeTimeStamp = 10;
    private static final String REGEX_IMAGE = "(http:|https:)(//)((?!\").)*?.(PNG|png|JPG|jpg|JPEG|jpeg)";
    private static final String REGEX_VIDEO = "(http:|https:)(//)((?!\").)*?.(mp4|MP4)";

    public static ConversationMsg getSupportReplyMsg(String response) {
        ConversationMsg conversationMsg = new ConversationMsg();
        conversationMsg.setMessageFromServer(true);
        conversationMsg.setMsgStatus(ConversationMsg.STATUS_SUCCESS);
        try {
            JSONObject pushReply = new JSONObject(response);
            if (pushReply.has("time")) {
                conversationMsg.setTimeStamp(pushReply.optLong("time"));
            }
            if (pushReply.has("msg")) {
                String msg = pushReply.optString("msg");
                conversationMsg.setMsgContent(msg);

                int msgType = ConversationMsg.TYPE_ADMIN_TEXT;
                if (Pattern.compile(REGEX_IMAGE).matcher(msg).matches()) {
                    msgType = ConversationMsg.TYPE_ADMIN_IMAGE;
                } else if (Pattern.compile(REGEX_IMAGE).matcher(msg).find()) {
                    msgType = ConversationMsg.TYPE_ADMIN_RICHTEXT;
                }

                conversationMsg.setMsgType(msgType);
            }
            if (pushReply.has("nickname")) {
                conversationMsg.setNickname(pushReply.optString("nickname"));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return conversationMsg;
    }

    public static ConversationMsg getUserTextMsg(boolean isBot, String msg) {
        ConversationMsg conversationMsg = new ConversationMsg();
        conversationMsg.setMsgType(isBot ? ConversationMsg.TYPE_USER_TEXT_BOT : ConversationMsg.TYPE_USER_TEXT);
        conversationMsg.setMsgStatus(isBot ? ConversationMsg.STATUS_SUCCESS : ConversationMsg.STATUS_SENDING);
        conversationMsg.setMsgContent(msg);
        return conversationMsg;
    }

    private static List<ChatHistoryEntity> getChatTimeStampList(JSONObject chatHistory) {
        ArrayList<ChatHistoryEntity> chatTimeStampList = new ArrayList<>();
        Pattern pattern = Pattern.compile("[0-9]*");

        Iterator<String> keys = chatHistory.keys();
        while (keys.hasNext()) {
            String key = keys.next();
            if (Const.TOGGLE_CHAT_TIMESTAMP && "chatTimestamp".equals(key)) {
                prepareHistoryTimeStamp(chatTimeStampList, chatHistory.optString("chatTimestamp"));
                continue;
            }

            String[] chatKey = key.split("\\|");
            if (pattern.matcher(chatKey[0]).matches()) {
                ChatHistoryEntity entity = new ChatHistoryEntity();
                entity.setTimeStamp(Long.parseLong(chatKey[0]));
                if (chatKey.length == 2) {
                    entity.setMsgType(ChatHistoryEntity.TYPE_ADMIN);
                    entity.setSupportName(chatKey[1]);
                    prepareSupportNickname(chatHistory, entity);
                } else {
                    entity.setMsgType(ChatHistoryEntity.TYPE_USER);
                    entity.setSupportName("");
                }

                chatTimeStampList.add(entity);
            }
        }
        Collections.sort(chatTimeStampList);
        return chatTimeStampList;
    }

    private static void prepareHistoryTimeStamp(ArrayList<ChatHistoryEntity> chatTimeStampList, String chatTimeStamp) {
        if (TextUtils.isEmpty(chatTimeStamp)) return;
        String[] timeStampArray = chatTimeStamp.split(",");
        for (String timeStamp : timeStampArray) {
            if (Pattern.compile("[0-9]*").matcher(timeStamp).matches()) {
                ChatHistoryEntity entity = new ChatHistoryEntity();
                entity.setMsgType(ChatHistoryEntity.TYPE_TIMESTAMP);
                entity.setTimeStamp(Long.parseLong(timeStamp) - 1);
                chatTimeStampList.add(entity);
            }
        }
    }

    private static void prepareSupportNickname(JSONObject chatHistory, ChatHistoryEntity entity) {
        if (chatHistory.has("assigneeinfo")) {
            try {
                JSONArray jsonArray = new JSONArray(chatHistory.optString("assigneeinfo"));
                if (jsonArray.length() > 0) {
                    for (int i = 0; i < jsonArray.length(); i++) {
                        JSONObject jsonObject = jsonArray.getJSONObject(i);
                        if (entity.getSupportName().equals(jsonObject.optString("username"))) {
                            String nickname = jsonObject.optString("nickname");
                            entity.setSupportNickname(nickname);
                            break;
                        }
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    private static String getConversationJSONObjectKey(ChatHistoryEntity historyEntity) {
        String key = String.valueOf(historyEntity.getTimeStamp());
        if (!TextUtils.isEmpty(historyEntity.getSupportName())) {
            key = key + "|" + historyEntity.getSupportName();
        }
        return key;
    }

    private static List<ConversationMsg> getSupportMsgAfterLogin(ChatHistoryEntity chatHistoryEntity, String msg) {

        ArrayList<ConversationMsg> loginMsgList = new ArrayList<>();
        final ConversationMsg conversationMsg = new ConversationMsg();
        conversationMsg.setTimeStamp(chatHistoryEntity.getTimeStamp());
        conversationMsg.setMessageFromServer(true);

        if (TextUtils.isEmpty(msg)) {
            if (chatHistoryEntity.getMsgType() == ChatHistoryEntity.TYPE_TIMESTAMP) {
                conversationMsg.setMsgType(ConversationMsg.TYPE_TIMESTAMP);
                conversationMsg.setMsgStatus(ConversationMsg.STATUS_SUCCESS);
                conversationMsg.setMsgContent(String.valueOf(chatHistoryEntity.getTimeStamp()));
                loginMsgList.add(conversationMsg);
            }
            return loginMsgList;
        }

        int msgType;
        String textMsg = msg;

        Matcher imageMatcher = Pattern.compile(REGEX_IMAGE).matcher(msg);
        Matcher videoMatcher = Pattern.compile(REGEX_VIDEO).matcher(msg);

        if (chatHistoryEntity.getMsgType() == ChatHistoryEntity.TYPE_ADMIN) {
            conversationMsg.setNickname(chatHistoryEntity.getSupportNickname());

            msgType = ConversationMsg.TYPE_ADMIN_TEXT;
            if (Pattern.compile(REGEX_IMAGE).matcher(msg).matches()) {
                msgType = ConversationMsg.TYPE_ADMIN_IMAGE;
            } else if (Pattern.compile(REGEX_IMAGE).matcher(msg).find()) {
                msgType = ConversationMsg.TYPE_ADMIN_RICHTEXT;
            }

        } else {
            if (imageMatcher.matches()) {
                msgType = ConversationMsg.TYPE_USER_IMAGE;
            } else if (videoMatcher.matches()) {
                msgType = ConversationMsg.TYPE_USER_VIDEO;
            } else {
                msgType = ConversationMsg.TYPE_USER_TEXT;

                while (imageMatcher.find()) {
                    textMsg = textMsg.replace(imageMatcher.group(), "");
                }

                Matcher newVideoMatcher = Pattern.compile(REGEX_VIDEO).matcher(textMsg);
                while (newVideoMatcher.find()) {
                    textMsg = textMsg.replace(newVideoMatcher.group(), "");
                }
            }
        }

        conversationMsg.setMsgType(msgType);
        conversationMsg.setMsgStatus(ConversationMsg.STATUS_SUCCESS);
        conversationMsg.setMsgContent(textMsg);
        loginMsgList.add(conversationMsg);

        // 提交表单生成的人工客诉，解析图片/视频附件
        if (conversationMsg.getMsgType() == ConversationMsg.TYPE_USER_TEXT) {
            replaceMediaUrls(msg, loginMsgList, conversationMsg);
        }

        return loginMsgList;
    }

    public static List<ConversationMsg> getUserFormMsgList(String msg) {

        ArrayList<ConversationMsg> formMsgList = new ArrayList<>();

        // 复制 msg 内容，用来获取表单的文字内容
        String textMsg = msg;
        Matcher imageMatcher = Pattern.compile(REGEX_IMAGE).matcher(msg);

        // 移除所有图片链接
        while (imageMatcher.find()) {
            textMsg = textMsg.replace(imageMatcher.group(), "");
        }

        // 使用移除图片链接后的字符串重新校验视频链接，否则会匹配中 imageUrl+videoUrl
        Matcher videoMatcher = Pattern.compile(REGEX_VIDEO).matcher(textMsg);
        while (videoMatcher.find()) {
            textMsg = textMsg.replace(videoMatcher.group(), "");
        }

        ConversationMsg conversationMsg = new ConversationMsg();
        conversationMsg.setMsgType(ConversationMsg.TYPE_USER_TEXT);
        conversationMsg.setMsgStatus(ConversationMsg.STATUS_SUCCESS);
        conversationMsg.setMsgContent(textMsg);
        formMsgList.add(conversationMsg);

        // 使用原始 msg 内容，提取所有 url 并转换为具体的图片或视频消息
        replaceMediaUrls(msg, formMsgList, conversationMsg);

        return formMsgList;
    }

    private static void replaceMediaUrls(String msg, ArrayList<ConversationMsg> formMsgList, ConversationMsg conversationMsg) {
        Matcher replaceImageMatcher = Pattern.compile(REGEX_IMAGE).matcher(msg);

        // 匹配图片并生成图片消息
        while (replaceImageMatcher.find()) {
            String imageUrl = replaceImageMatcher.group();
            ConversationMsg formImageMsg = new ConversationMsg();
            formImageMsg.setTimeStamp(conversationMsg.getTimeStamp() + sFakeTimeStamp++);
            formImageMsg.setMessageFromServer(true);
            formImageMsg.setMsgType(ConversationMsg.TYPE_USER_IMAGE);
            formImageMsg.setMsgStatus(ConversationMsg.STATUS_SUCCESS);
            formImageMsg.setMsgContent(imageUrl);
            formMsgList.add(formImageMsg);
            // 添加消息成功后就移除对应的链接
            msg = msg.replace(imageUrl, "");
        }

        // 使用移除 image 后的字符串重新尝试匹配 videoUrl
        Matcher replaceVideoMatcher = Pattern.compile(REGEX_VIDEO).matcher(msg);
        while (replaceVideoMatcher.find()) {
            String videoUrl = replaceVideoMatcher.group();
            ConversationMsg formVideoMsg = new ConversationMsg();
            formVideoMsg.setTimeStamp(conversationMsg.getTimeStamp() + sFakeTimeStamp++);
            formVideoMsg.setMessageFromServer(true);
            formVideoMsg.setMsgType(ConversationMsg.TYPE_USER_VIDEO);
            formVideoMsg.setMsgStatus(ConversationMsg.STATUS_SUCCESS);
            formVideoMsg.setMsgContent(videoUrl);
            formMsgList.add(formVideoMsg);
            msg = msg.replace(videoUrl, "");
        }
    }

    public static List<ConversationMsg> getRetrievedMsgList(JSONObject msgJson) {
        ArrayList<ConversationMsg> list = new ArrayList<>();
        try {
            if (msgJson != null && msgJson.length() > 0) {
                for (ChatHistoryEntity historyEntity : ConversationHelper.getChatTimeStampList(msgJson)) {
                    // 遍历排好序的时间，然后按时间顺序取出消息
                    // historyEntity 里有 type，根据 type 判断是客服还是用户，然后生成消息对象
                    String chatMsg = msgJson.optString(ConversationHelper.getConversationJSONObjectKey(historyEntity));
                    list.addAll(ConversationHelper.getSupportMsgAfterLogin(historyEntity, chatMsg));
                }
            }
        } catch (Exception e) {
            TLog.e("getRetrievedMsgList, Exception " + e.toString());
        }
        return list;
    }

    public static ElvaBotMsg getDefaultMsg() {
        return ElvaBotHelper.getDefaultReply();
    }

    public static ElvaBotMsg getBotReply(String input) {
        return ElvaBotHelper.getLocalReply(input);
    }

    public static long getWithdrawTimeStamp(String response) {
        try {
            JSONObject pushWithdraw = new JSONObject(response);
            if (pushWithdraw.has("withdrawkey")) {
                return pushWithdraw.getLong("withdrawkey");
            }
        } catch (Exception e) {
            e.printStackTrace();
            return 0;
        }
        return 0;
    }

}
