package net.aihelp.ui.cs;

import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.graphics.Color;
import android.os.Bundle;
import android.os.Message;
import android.text.TextUtils;
import android.view.View;
import android.view.animation.LinearInterpolator;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import net.aihelp.R;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.ui.dialog.AlertDialog;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.concurrent.ApiExecutor;
import net.aihelp.core.util.concurrent.ApiExecutorFactory;
import net.aihelp.data.event.SupportActionEvent;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.data.logic.ElvaBotPresenter;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.data.model.cs.ElvaBotMsg;
import net.aihelp.db.AIHelpDBHelper;
import net.aihelp.db.faq.pojo.DisplayFaq;
import net.aihelp.ui.adapter.FaqAlertAdapter;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.adapter.TextWatcherAdapter;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.LogoutMqttHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.TLog;

import java.util.ArrayList;
import java.util.List;

import androidx.appcompat.widget.AppCompatImageView;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class ElvaBotFragment extends BaseCSFragment<ElvaBotPresenter> {

    private RecyclerView rvFaqHintList;
    private FaqAlertAdapter mFaqAlertAdapter;
    private LinearLayout llChecking;
    private AppCompatImageView ivChecking;

    public static ElvaBotFragment newInstance(Bundle bundle) {
        ElvaBotFragment elvaBot = new ElvaBotFragment();
        elvaBot.setArguments(bundle);
        return elvaBot;
    }

    @Override
    public void onResume() {
        super.onResume();

        if (isOperateBot) {
            if (!MqttConfig.getInstance().isConnected()) {
                mPresenter.prepareMqttConnection(mqttCallback);
            }
            return;
        }
        ResponseMqttHelper.updateElvaSupportActionStatus(isAlwaysShowSupportInElva);
    }

    @Override
    protected void initEventAndData(View contentView) {
        super.initEventAndData(contentView);

        StatisticHelper.whenBotVisible();

        llChecking = get(R.id.aihelp_ll_net_checking);
        ivChecking = get(R.id.aihelp_iv_checking);
        rvFaqHintList = get(R.id.aihelp_rv_faq_alert);
        rvFaqHintList.setLayoutManager(new LinearLayoutManager(getContext()));
        mFaqAlertAdapter = new FaqAlertAdapter(getContext(), new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mFaqAlertAdapter.clear();
                whenUserSay(((TextView) v).getText().toString());
                StatisticHelper.whenSendButtonClicked();
            }
        });
        rvFaqHintList.setAdapter(mFaqAlertAdapter);

        mAdapter.setOnClickedListener(new MessageListAdapter.OnClickedListenerWrapper() {

            @Override
            public void onActionClicked(String action) {
                whenUserSay(action);
            }

            @Override
            public void onUrlClicked(String url) {
                if (mPresenter.validateNetwork()) {
                    Bundle bundle = new Bundle();
                    bundle.putString(IntentValues.INTENT_URL, url);
                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                }
            }

            @Override
            public void onFormUrlClicked(long timeStamp, String formUrl) {
                if (!MqttConfig.getInstance().isConnected()) {
                    mPresenter.prepareMqttConnection(mqttCallback);
                }
                if (mPresenter.validateNetwork()) {
                    Bundle bundle = new Bundle();
                    bundle.putString(IntentValues.INTENT_URL, formUrl);
                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                    LogoutMqttHelper.updateType(LogoutMqttHelper.LOGOUT_TYPE_FORM_GOTO_PAGE);
                    StatisticHelper.whenFormEventHappened(timeStamp, StatisticHelper.FORM_ACTION_CLICKED);
                    syncLogoutTypeToServer();
                }
            }

            @Override
            public void onFaqHelpfulClicked(boolean helpful, long timeStamp) {
                mPresenter.markWhetherFaqHelpful(helpful, timeStamp);
                LogoutMqttHelper.updateType(helpful ? LogoutMqttHelper.LOGOUT_TYPE_FAQ_HELPFUL
                        : LogoutMqttHelper.LOGOUT_TYPE_FAQ_UNHELPFUL);
            }

            @Override
            public void onFaqFeedbackClicked(TextView view, ElvaBotMsg botMsg, String faqId) {
                showAdviceAlert(view, botMsg, faqId);
            }

            @Override
            public void onFaqClicked(ElvaBotMsg botMsg) {
                if (mPresenter.validateNetwork()) {
                    if (botMsg != null) {
                        TLog.e("faqUrl: " + botMsg.getBotUrl().getUrlAddress());
                        new ElvaFaqBottomSheetEvent(ElvaBotFragment.this).show(botMsg.getBotUrl().getUrlAddress());
                    }
                }
            }

        });

        if (!isOperateBot) {
            mPresenter.prepareMqttConnection(mqttCallback);
        }

    }

    private void showAdviceAlert(final TextView view, final ElvaBotMsg botMsg, final String faqContentId) {
        final AlertDialog mFeedbackDialog = new AlertDialog.Builder(getContext())
                .setContentView(R.layout.aihelp_dia_advice)
                .setCancelableOntheOutside(true)
                .setWidthByDevice()
                .create();

        final TextView tvConfirm = mFeedbackDialog.getView(R.id.aihelp_tv_confirm);
        final EditText etFeedback = mFeedbackDialog.getView(R.id.aihelp_et_feedback);
        etFeedback.addTextChangedListener(new TextWatcherAdapter() {
            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                tvConfirm.setEnabled(!TextUtils.isEmpty(s.toString().trim()));
                tvConfirm.setAlpha(TextUtils.isEmpty(s.toString().trim()) ? 0.5f : 1f);
            }
        });

        mFeedbackDialog.setOnClickListener(R.id.aihelp_tv_cancel, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mFeedbackDialog.dismiss();
            }
        });
        mFeedbackDialog.setOnClickListener(R.id.aihelp_tv_confirm, new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (TextUtils.isEmpty(etFeedback.getText().toString().trim())) {
                    Toast.makeText(getContext(), getResources().getString(R.string.aihelp_faq_feedback), Toast.LENGTH_SHORT).show();
                    return;
                }
                mPresenter.postFeedbackOnFaq(botMsg, faqContentId, etFeedback.getText().toString().trim());
                mFeedbackDialog.dismiss();
                SoftInputUtil.hideSoftInput(getContext(), etFeedback);
                view.setEnabled(false);
                view.setTextColor(Color.parseColor("#666666"));
            }
        });
        mFeedbackDialog.show();
    }

    @Override
    protected void onTextChange(final CharSequence s) {

        if (TextUtils.isEmpty(s.toString().trim())) {
            rvFaqHintList.setVisibility(View.GONE);
            mFaqAlertAdapter.clear();
            return;
        }

        final ApiExecutor apiExecutor = ApiExecutorFactory.getHandlerExecutor();
        apiExecutor.runAsync(new Runnable() {
            @Override
            public void run() {
                final ArrayList<DisplayFaq> faqs = AIHelpDBHelper.getInstance().getMatchedFaqListForAlert(String.valueOf(s));
                apiExecutor.runOnUiThread(new Runnable() {
                    @Override
                    public void run() {
                        rvFaqHintList.setVisibility(faqs.size() > 0 ? View.VISIBLE : View.GONE);
                        mFaqAlertAdapter.update(faqs, true);
                    }
                });
            }
        });
    }

    @Override
    protected int getLayout() {
        return R.layout.aihelp_fra_elva_bot;
    }

    @Override
    protected int getLoadingTargetViewId() {
        return R.id.aihelp_elva_root;
    }

    @Override
    public void onClick(View v) {
        if (v.getId() == R.id.aihelp_btn_send) {
            StatisticHelper.whenSendButtonClicked();
            String input = etInput.getText().toString();
            whenUserSay(input);
        }
    }

    private void whenUserSay(String userSay) {
        updateChatList(ConversationHelper.getUserTextMsg(true, userSay));
        Message message = Message.obtain();
        message.obj = userSay;
        mHandler.sendMessage(message);
    }

    @Override
    protected void handleMsg(Message msg) {

        String userInput = (String) msg.obj;
        final ElvaBotMsg botReply = ConversationHelper.getBotReply(userInput);
        if (!botReply.isBotStupid() || !mPresenter.isNetworkAvailable()) {
            rvMsgList.postDelayed(new Runnable() {
                @Override
                public void run() {
                    updateChatList(botReply);
                }
            }, 800);
        } else {
            mPresenter.askForAnswer(userInput);
        }

    }

    @Override
    public void updateChatList(ConversationMsg msg) {
        super.updateChatList(msg);

        // 对每组机器人问答进行打点统计，同时更新标签信息，用于同步机器人客诉
        if (msg instanceof ElvaBotMsg) {
            ElvaBotMsg botMsg = (ElvaBotMsg) msg;
            mPresenter.handleBotMsg(botMsg);
        }

    }

    public void updateFAQFeedback(String timeMillis, boolean isLike, String ticketId) {
        for (int i = 0; i < mAdapter.getDataList().size(); i++) {
            ConversationMsg conversationMsg = mAdapter.getDataList().get(i);
            if (conversationMsg.getMsgType() == ConversationMsg.TYPE_ADMIN_FAQ) {
                if (String.valueOf(conversationMsg.getTimeStamp()).equals(timeMillis)) {
                    conversationMsg.setMsgStatus(isLike ? ConversationMsg.STATUS_FAQ_HELPFUL : ConversationMsg.STATUS_FAQ_UNHELPFUL);
                    conversationMsg.setFaqTicketId(ticketId);
                    mAdapter.notifyItemChanged(i);
                }
            }
        }
    }

    @Override
    public void onMqttLogin(List<ConversationMsg> mqttReplyMsg) {
        super.onMqttLogin(mqttReplyMsg);
        if (isVisible() && ResponseMqttHelper.isTicketUnFinish()) {
            EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_CONVERSATION));
        }
    }

    public void markSupportActionUnread() {
        // flag for previous page to show unread status
        ResponseMqttHelper.setHasUnreadMsg(true);

        // To avoid showing CS entrance while filling form when there is a response
        if (isVisible()) {
            EventBus.getDefault().post(new SupportActionEvent(IntentValues.SHOW_SUPPORT_ACTION,
                    IntentValues.SUPPORT_ACTION_MSG_UNREAD));
        }
    }

    public void syncLogoutTypeToServer() {

    }

    @Override
    public void updateNetCheckingStatus(boolean startChecking) {
        if (startChecking) {
            llChecking.setVisibility(View.VISIBLE);
            ObjectAnimator loadingAnimator = ObjectAnimator.ofFloat(ivChecking, "rotation", 0F, 360F);
            loadingAnimator.setDuration(1000);
            loadingAnimator.setInterpolator(new LinearInterpolator());
            loadingAnimator.setRepeatCount(ValueAnimator.INFINITE);
            loadingAnimator.start();
        } else {
            llChecking.setVisibility(View.GONE);
        }
    }

    @Override
    public boolean isNetCheckingInProgress() {
        return llChecking.getVisibility() == View.VISIBLE;
    }

    public boolean onBackPressed() {
        if (isNetCheckingInProgress()) {
            showNetworkInterruptDialog();
            return false;
        }
        return true;
    }

    private void showNetworkInterruptDialog() {
        if (getActivity() != null) {
            final AlertDialog netCheckDialog = new AlertDialog.Builder(getContext())
                    .setContentView(R.layout.aihelp_dia_interrupt_net_check)
                    .setWidthByDevice()
                    .create();

            netCheckDialog.setOnClickListener(R.id.aihelp_tv_cancel, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    netCheckDialog.dismiss();
                }
            });
            netCheckDialog.setOnClickListener(R.id.aihelp_tv_confirm, new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    netCheckDialog.dismiss();
                    llChecking.setVisibility(View.GONE);
                    if (getActivity() != null) getActivity().onBackPressed();
                }
            });

            if (getActivity() != null && !getActivity().isFinishing()) {
                netCheckDialog.show();
            } else {
                TLog.e("Activity finished when network dialog is about to show.");
            }
        }

    }

    @Override
    public void onNetworkStateChanged(NetworkState networkState) {
        if (networkState == NetworkState.NONE) {
            mqttCallback.onMqttFailure(null);
        } else {
            mPresenter.prepareMqttConnection(mqttCallback);
        }
    }

    @Override
    public void onDestroy() {
        if (mPresenter != null) mPresenter.logoutMqtt(false);
        super.onDestroy();
    }

}
