package net.aihelp.ui.cs;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.os.Bundle;
import android.os.Message;
import android.text.TextUtils;
import android.view.View;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.Toast;

import net.aihelp.R;
import net.aihelp.common.Const;
import net.aihelp.common.IntentValues;
import net.aihelp.core.net.monitor.NetworkMonitor;
import net.aihelp.core.net.monitor.NetworkState;
import net.aihelp.core.net.mqtt.config.MqttConfig;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.luban.Luban;
import net.aihelp.core.util.luban.OnCompressListener;
import net.aihelp.core.util.permission.AIHelpPermissions;
import net.aihelp.core.util.permission.IPermissionCallback;
import net.aihelp.core.util.permission.Permission;
import net.aihelp.core.util.permission.PermissionHelper;
import net.aihelp.data.event.PageHoppingEvent;
import net.aihelp.data.event.PrepareMessageTimeStampEvent;
import net.aihelp.data.logic.ConversationPresenter;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.helper.AttachmentPicker;
import net.aihelp.ui.helper.AttachmentPickerFile;
import net.aihelp.ui.helper.BitmapHelper;
import net.aihelp.ui.helper.ConversationHelper;
import net.aihelp.ui.helper.EvaluateHelper;
import net.aihelp.ui.helper.ResponseMqttHelper;
import net.aihelp.ui.helper.SkinHelper;
import net.aihelp.ui.helper.StatisticHelper;
import net.aihelp.ui.widget.AIHelpSwipeRefreshLayout;
import net.aihelp.ui.widget.snackbar.Snackbar;
import net.aihelp.utils.ListUtil;
import net.aihelp.utils.MediaUtils;
import net.aihelp.utils.PathUtils;
import net.aihelp.utils.Styles;
import net.aihelp.utils.TLog;
import net.aihelp.utils.ToastUtil;

import org.json.JSONArray;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.regex.Pattern;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class ConversationFragment extends BaseCSFragment<ConversationPresenter> implements AttachmentPicker.AttachmentPickerListener {

    private boolean isModeConversation;
    private AIHelpSwipeRefreshLayout mRefreshLayout;
    private int mCurrentPage = 1;
    private boolean isTicketAlreadyFinished;

    private LinearLayout llInput;
    private LinearLayout llRate;
    private boolean isRateApp;

    public static ConversationFragment newInstance(Bundle bundle) {
        ConversationFragment fraConversation = new ConversationFragment();
        fraConversation.setArguments(bundle);
        return fraConversation;
    }

    @Override
    protected void initEventAndData(View contentView) {
        super.initEventAndData(contentView);

        llInput = get(R.id.aihelp_ll_input);
        llRate = get(R.id.aihelp_ll_rate);
        mRefreshLayout = get(R.id.aihelp_refresh_layout);
        etInput.setText(mPresenter.getInputDraft());

        mAdapter.setOnClickedListener(new MessageListAdapter.OnClickedListenerWrapper() {

            @Override
            public void onUrlClicked(String url) {
                if (mPresenter.validateNetwork()) {
                    Bundle bundle = new Bundle();
                    bundle.putString(IntentValues.INTENT_URL, url);
                    EventBus.getDefault().post(new PageHoppingEvent(IntentValues.PAGE_HOPPING_FORM, bundle));
                }
            }

            @Override
            public void onRetrySendingMessage(int position, ConversationMsg msg) {

                if (!mPresenter.isNetworkAvailable() || !MqttConfig.getInstance().isConnected()) {
                    Toast.makeText(getContext(), getString(R.string.aihelp_network_no_connect), Toast.LENGTH_SHORT).show();
                    return;
                }

                if (isTicketAlreadyFinished) {
                    Toast.makeText(getContext(), getString(R.string.aihelp_ticket_closed), Toast.LENGTH_SHORT).show();
                    return;
                }

                mAdapter.remove(position);
                mAdapter.notifyItemChanged(position);

                if (msg.getMsgType() == ConversationMsg.TYPE_USER_TEXT) {
                    msg.setTimeStamp(System.currentTimeMillis());
                    updateChatList(msg);
                    chatWithSupportWithTimeout(msg.getTimeStamp(), msg.getMsgContent());
                }

                if (msg.getMsgType() == ConversationMsg.TYPE_USER_IMAGE) {
                    uploadImage(msg.getMsgContent());
                }

                if (msg.getMsgType() == ConversationMsg.TYPE_USER_VIDEO) {
                    uploadVideo(msg.getMsgContent());
                }

            }

        });

        mRefreshLayout.setEnabled(Const.TOGGLE_GET_LAST_CONVERSATION);
        mRefreshLayout.setColorSchemeResources(R.color.aihelp_color_main);
        if (Const.TOGGLE_GET_LAST_CONVERSATION) {
            mRefreshLayout.setOnRefreshListener(new AIHelpSwipeRefreshLayout.OnRefreshListener() {
                @Override
                public void onRefresh() {
                    mPresenter.getLastConversation(mCurrentPage);
                }
            });
        }

        if (PermissionHelper.hasReadPermission(getContext())) {
            get(R.id.aihelp_iv_attach).setVisibility(View.VISIBLE);
            SkinHelper.updateIcon(SkinHelper.SKIN_UPLOAD_MEDIA, get(R.id.aihelp_iv_attach));
            get(R.id.aihelp_iv_attach).setOnClickListener(this);
        } else {
            get(R.id.aihelp_iv_attach).setVisibility(View.GONE);
            LinearLayout.LayoutParams etInputParams = (LinearLayout.LayoutParams) etInput.getLayoutParams();
            etInputParams.leftMargin = (int) Styles.dpToPx(getContext(), 15);
            etInput.setLayoutParams(etInputParams);
        }

        get(R.id.aihelp_btn_go_rate).setOnClickListener(this);
        ResponseMqttHelper.resetFlagsAfterMsgViewed();

        if (Const.TOGGLE_OPEN_UNREAD_MSG && Const.sMessageListener != null) {
            Const.sMessageListener.onMessageCountArrived(0);
        }

        if (!isOperateHuman) {
            mPresenter.prepareMqttConnection(mqttCallback);
        }

    }

    @Override
    protected void getBundleAfterDataPrepared(Bundle extras) {
        isModeConversation = IntentValues.MODE_CONVERSATION == extras.getInt(IntentValues.SUPPORT_MODE);
    }

    @Override
    public void onResume() {
        super.onResume();

        if (!MqttConfig.getInstance().isConnected() && isOperateHuman) {
            mPresenter.prepareMqttConnection(mqttCallback);
            return;
        }

        // After form submitted, scroll the message list to the end to display form content
        if (ResponseMqttHelper.isFormSubmitted()) {
            ResponseMqttHelper.setFormSubmitStatus(false);
            rvMsgList.scrollToPosition(mAdapter.getItemCount() - 1);
        }

    }

    @Override
    public void onDestroy() {
        if (mPresenter != null && mAdapter != null) {
            mPresenter.saveInputDraftIfNeeded(etInput.getText().toString().trim());
            if (isOperateHuman || isModeConversation) mPresenter.logoutMqtt(true);
        }
        super.onDestroy();
    }

    @Override
    protected int getLayout() {
        return R.layout.aihelp_fra_conversation;
    }

    @Override
    protected int getLoadingTargetViewId() {
        return R.id.aihelp_conversation_root;
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        AIHelpPermissions.getInstance().onRequestPermissionsResult(grantResults.length == 1 && grantResults[0] == 0);
    }

    @Permission(requestCode = 1000)
    public void launchPicker(Permission.Result result, final IPermissionCallback callback) {
        switch (result) {
            case GRANTED:
                AttachmentPicker.getInstance().setPickerHost(this).launchImagePicker(true);
                break;
            case DENIED:
                ToastUtil.showRawSnackBar(getActivity(), getString(R.string.aihelp_permission_denied), Snackbar.LENGTH_LONG);
                break;
            case RATIONAL:
                ToastUtil.showRawSnackBar(getActivity(), getString(R.string.aihelp_permission_denied),
                        getString(R.string.aihelp_yes), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                callback.onPermissionRational();
                            }
                        });
                break;
            case GO_SETTING:
                ToastUtil.showRawSnackBar(getActivity(), getString(R.string.aihelp_permission_ignored),
                        getString(R.string.aihelp_permission_settings), Snackbar.LENGTH_LONG, new View.OnClickListener() {
                            @Override
                            public void onClick(View v) {
                                callback.onPermissionIgnored();
                            }
                        });
                break;
        }
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent intent) {
        super.onActivityResult(requestCode, resultCode, intent);
        if (intent != null && resultCode == Activity.RESULT_OK) {
            AttachmentPicker.getInstance().onAttachmentPickRequestResult(requestCode, intent);
        }
    }

    private void chatWithSupportWithTimeout(long timeStamp, String input) {
        mPresenter.chatWithSupport(timeStamp, input);
        sendDelayTimeoutMsg(timeStamp, 20);
    }

    public void onMqttLogin(List<ConversationMsg> mqttReplyMsg) {
        if (mAdapter.getItemCount() > 0) {
            List<ConversationMsg> exceptionList = getListIfDisconnectionEverHappened();
            if (!ListUtil.isListEmpty(exceptionList)) {
                mqttReplyMsg.addAll(exceptionList);
                Collections.sort(mqttReplyMsg, new Comparator<ConversationMsg>() {
                    @Override
                    public int compare(ConversationMsg o1, ConversationMsg o2) {
                        Long o1Time = o1.getTimeStamp();
                        Long o2Time = o2.getTimeStamp();
                        return o1Time.compareTo(o2Time);
                    }
                });
                for (ConversationMsg msg : exceptionList) {
                    if (msg.getMsgStatus() == ConversationMsg.STATUS_SENDING) {
                        if (msg.getMsgType() == ConversationMsg.TYPE_USER_IMAGE) {
                            mPresenter.uploadImage(new File(msg.getMsgContent()), msg.getTimeStamp());
                        } else if (msg.getMsgType() == ConversationMsg.TYPE_USER_VIDEO) {
                            mPresenter.uploadVideo(new File(msg.getMsgContent()), msg.getTimeStamp());
                        } else {
                            chatWithSupportWithTimeout(msg.getTimeStamp(), msg.getMsgContent());
                        }
                    }
                }
            }
        }
        // reset current page, in case of reconnecting after network disconnect.
        mCurrentPage = 1;
        super.onMqttLogin(mqttReplyMsg);

        if (isVisible() && ResponseMqttHelper.isTicketUnRated()) {
            onConversationFinished(false);
        }
    }

    private List<ConversationMsg> getListIfDisconnectionEverHappened() {
        List<ConversationMsg> failedList = new ArrayList<>();
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            ConversationMsg msg = mAdapter.getDataList().get(i);
            if (msg.getMsgStatus() != ConversationMsg.STATUS_SUCCESS) {
                failedList.add(0, msg);
            }
        }
        return failedList;
    }

    public void onConversationFinished(boolean isRateApp) {

        isTicketAlreadyFinished = true;
        llInput.postDelayed(new Runnable() {
            @Override
            public void run() {
                llInput.setVisibility(View.GONE);
                llRate.setVisibility(View.VISIBLE);
            }
        }, 500);

        if (!isVisible() || EvaluateHelper.getInstance().isShowing()) return;

        this.isRateApp = isRateApp;
        EvaluateHelper.getInstance().showRateSupport(getContext(), isRateApp, new EvaluateHelper.OnConfirmEvaluateListener() {
            @Override
            public boolean onConfirmEvaluate(int rate, String input, JSONArray selected) {
                if (!mPresenter.isNetworkAvailable() && !MqttConfig.getInstance().isConnected()) {
                    Toast.makeText(getContext(), getString(R.string.aihelp_network_no_connect), Toast.LENGTH_SHORT).show();
                    return false;
                }
                bottomLayout.setVisibility(View.GONE);
                mPresenter.evaluateSupport(rate, input, selected);
                ResponseMqttHelper.setTicketUnFinish(false);
                mRefreshLayout.setEnabled(false);
                return true;
            }
        });

    }

    @Override
    public void onPickSuccess(AttachmentPickerFile file, Bundle bundle) {
        if (isTicketAlreadyFinished) return;

        final String path = PathUtils.getPath(getContext(), file.transientUri);
        if (TextUtils.isEmpty(path)) {
            ToastUtil.makeText(getContext(), "Failed to get file path.", false);
            return;
        }

        if (Pattern.compile("^/.+\\.(png|jpg|jpeg|PNG|JPG|JPEG)$").matcher(path).matches()) {
            uploadImage(path);
        }

        if (Pattern.compile("^/.+\\.(mp4|MP4)$").matcher(path).matches()) {
            uploadVideo(path);
        }

    }

    private void uploadVideo(final String path) {
        MediaUtils.getImageForVideo(path, new MediaUtils.OnLoadVideoImageListener() {
            @Override
            public void onLoadImage(File file) {
                final ConversationMsg videoMsg = new ConversationMsg(ConversationMsg.TYPE_USER_VIDEO, ConversationMsg.STATUS_SENDING);
                videoMsg.setMsgContent(path);
                updateChatList(videoMsg);
                mPresenter.uploadVideo(new File(path), videoMsg.getTimeStamp());
                sendDelayTimeoutMsg(videoMsg.getTimeStamp(), 150);
            }
        });

    }

    private void uploadImage(final String path) {
        Luban.with(getContext()).load(path).setCompressListener(new OnCompressListener() {
            @Override
            public void onStart() {
            }

            @Override
            public void onSuccess(File compressedFile) {
                uploadAfterCompress(compressedFile);
            }

            @Override
            public void onError(Throwable e) {
                uploadAfterCompress(new File(path));
            }

            private void uploadAfterCompress(File file) {
                ConversationMsg imageMsg = new ConversationMsg(ConversationMsg.TYPE_USER_IMAGE, ConversationMsg.STATUS_SENDING);
                imageMsg.setImageSize(BitmapHelper.computeSize(file.getPath()));
                imageMsg.setMsgContent(file.getPath());
                updateChatList(imageMsg);
                mPresenter.uploadImage(file, imageMsg.getTimeStamp());
                sendDelayTimeoutMsg(imageMsg.getTimeStamp(), 50);
            }

        }).launch();
    }

    @SuppressLint("StringFormatInvalid")
    @Override
    public void onPickFailure(int code, Long limitSize) {
        switch (code) {
            case AttachmentPicker.ATTACHMENT_FILE_SIZE_LIMIT_EXCEEDED:
                String msg = getResources().getString(R.string.aihelp_media_upload_err_size);
                try {
                    Toast.makeText(getContext(), String.format(msg, String.valueOf(limitSize / 1048576)), Toast.LENGTH_LONG).show();
                } catch (Exception e) {
                    e.printStackTrace();
                    Toast.makeText(getContext(), msg + ", < " + (limitSize / 1048576) + "M", Toast.LENGTH_LONG).show();
                }
                break;
            case AttachmentPicker.INVALID_URI:
                Toast.makeText(getContext(), "Failed to get specific resource", Toast.LENGTH_LONG).show();
                break;
        }
    }

    public void withdrawMsg(long withdrawTimeStamp) {
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            ConversationMsg msg = mAdapter.getDataList().get(i);
            if (msg.getTimeStamp() == withdrawTimeStamp) {
                mAdapter.getDataList().remove(i);
                mAdapter.notifyDataSetChanged();
                mPresenter.updateCachedUnreadMessageCount(true);
                break;
            }
        }
    }

    @NetworkMonitor
    public void onNetworkStateChanged(NetworkState networkState) {
        if (networkState == NetworkState.NONE) {
            mqttCallback.onMqttFailure(null);
        } else {
            // If this page comes from bot, the mqtt connect should be requested by that page, not here.
            if (isModeConversation) {
                mPresenter.prepareMqttConnection(mqttCallback);
            }
        }
    }

    public void onLastConversationRetrieved(List<ConversationMsg> msgList) {
        mRefreshLayout.setRefreshing(false);
        if (msgList != null) {
            mCurrentPage++;
            mAdapter.insertHistoryConversation(msgList);
            RecyclerView.LayoutManager layoutManager = rvMsgList.getLayoutManager();
            if (layoutManager instanceof LinearLayoutManager) {
                LinearLayoutManager linearManager = (LinearLayoutManager) layoutManager;
                int lastItemPosition = linearManager.findLastVisibleItemPosition();
                rvMsgList.scrollToPosition(msgList.size() - 1 + lastItemPosition);
                rvMsgList.smoothScrollBy(0, -100);
            }
        }
    }

    public void onFormSubmitted(String formContent) {
        mAdapter.update(ConversationHelper.getUserFormMsgList(formContent), false);
        rvMsgList.scrollToPosition(mAdapter.getItemCount() - 1);
    }

    private void sendDelayTimeoutMsg(long timeStamp, long delaySeconds) {
        Message message = Message.obtain();
        message.obj = timeStamp;
        message.what = getMessageWhatFromTimeStamp(String.valueOf(timeStamp));
        mHandler.sendMessageDelayed(message, delaySeconds * 1000);
    }

    protected void handleMsg(Message msg) {
        updateMsgStatus(false, (Long) msg.obj);
    }

    private int getMessageWhatFromTimeStamp(String timeStamp) {
        return Integer.parseInt(timeStamp.substring(timeStamp.length() / 5 * 2));
    }

    public void updateMsgStatus(boolean isSuccess, long timeStamp) {
        mHandler.removeMessages(getMessageWhatFromTimeStamp(String.valueOf(timeStamp)));
        for (int i = mAdapter.getDataList().size() - 1; i >= 0; i--) {
            ConversationMsg msg = mAdapter.getDataList().get(i);
            if (msg.getTimeStamp() == timeStamp) {
                msg.setMsgStatus(isSuccess ? ConversationMsg.STATUS_SUCCESS : ConversationMsg.STATUS_RETRY);
                mAdapter.notifyItemChanged(i);
                break;
            }
        }
    }

    @Override
    public void onEventComing(PrepareMessageTimeStampEvent event) {
        super.onEventComing(event);
        mPresenter.updateCachedUnreadMessageCount(false);
    }

    @Override
    public void onClick(View v) {
        if (v.getId() == R.id.aihelp_btn_send) {
            if (mPresenter.validateNetwork()) {
                StatisticHelper.whenSendButtonClicked();
                String input = etInput.getText().toString().trim();
                mPresenter.clearInputDraft();
                ConversationMsg msg = ConversationHelper.getUserTextMsg(false, input);
                updateChatList(msg);
                chatWithSupportWithTimeout(msg.getTimeStamp(), input);
            }
        }

        if (v.getId() == R.id.aihelp_iv_attach) {
            if (mPresenter.validateNetwork()) {
                AIHelpPermissions.getInstance().setHost(ConversationFragment.this).setRequestCode(1000)
                        .setRequestPermission(Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN ?
                                Manifest.permission.READ_EXTERNAL_STORAGE : "")
                        .request();
            }
        }

        if (v.getId() == R.id.aihelp_btn_go_rate) {
            onConversationFinished(isRateApp);
        }
    }

}
