package net.aihelp.ui.cs;

import android.content.res.Configuration;
import android.graphics.Color;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;

import net.aihelp.R;
import net.aihelp.common.Const;
import net.aihelp.common.IntentValues;
import net.aihelp.core.mvp.IPresenter;
import net.aihelp.core.net.mqtt.callback.IMqttCallback;
import net.aihelp.core.ui.BaseFragment;
import net.aihelp.core.util.bus.EventBus;
import net.aihelp.core.util.bus.Subscribe;
import net.aihelp.core.util.bus.ThreadMode;
import net.aihelp.data.event.PrepareMessageTimeStampEvent;
import net.aihelp.data.event.SkinUpdateEvent;
import net.aihelp.data.event.SupportActionEvent;
import net.aihelp.data.logic.MqttCallbackImpl;
import net.aihelp.data.model.cs.ConversationMsg;
import net.aihelp.ui.adapter.MessageListAdapter;
import net.aihelp.ui.adapter.TextWatcherAdapter;
import net.aihelp.ui.helper.SkinHelper;
import net.aihelp.ui.widget.AIHelpRecyclerView;
import net.aihelp.utils.KeyboardChangeHelper;
import net.aihelp.utils.SoftInputUtil;
import net.aihelp.utils.TLog;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.appcompat.widget.AppCompatImageButton;
import androidx.core.graphics.drawable.DrawableCompat;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.vectordrawable.graphics.drawable.VectorDrawableCompat;


public abstract class BaseCSFragment<P extends IPresenter> extends BaseFragment<P> implements View.OnClickListener {

    AIHelpRecyclerView rvMsgList;
    MessageListAdapter mAdapter;

    ViewGroup bottomLayout;
    EditText etInput;
    AppCompatImageButton btnSend;
    IMqttCallback mqttCallback;

    boolean isAlwaysShowSupportInElva;
    private KeyboardChangeHelper helper;
    boolean isOperateBot;
    boolean isOperateHuman;
    boolean isKeyboardShowing;

    private ConversationMsg mTimeStampMsg;

    @Override
    protected void getBundleBeforeDataPrepared(Bundle extras) {
        isOperateBot = extras.getBoolean(IntentValues.BOT_EMBEDDED_IN_OPERATION, false);
        isOperateHuman = extras.getBoolean(IntentValues.HUMAN_EMBEDDED_IN_OPERATION, false);
    }

    @Override
    public void onResume() {
        super.onResume();
        if (!isOperateBot || !isOperateHuman) {
            // 运营模块标题不显示皮肤，只有正常 showConversation 打开的页面才显示皮肤
            EventBus.getDefault().post(new SkinUpdateEvent(IntentValues.SHOW_NAVIGATION_SKIN));
        }
    }

    @Override
    protected void initEventAndData(View contentView) {
        EventBus.getDefault().post(new SupportActionEvent(IntentValues.HIDE_SUPPORT_ACTION));

        mqttCallback = MqttCallbackImpl.getInstance();
        bottomLayout = get(R.id.aihelp_chat_bottom_layout);
        etInput = get(R.id.aihelp_et_input);
        btnSend = get(R.id.aihelp_btn_send);
        rvMsgList = get(R.id.aihelp_rv_msg_list);

        mAdapter = new MessageListAdapter(getContext());
        rvMsgList.setAdapter(mAdapter);

        final LinearLayoutManager linearLayoutManager = new LinearLayoutManager(getContext());
        rvMsgList.setLayoutManager(linearLayoutManager);

        if (this.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
            SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BACKGROUND_PORTRAIT, rvMsgList);
        } else {
            SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BACKGROUND_LANDSCAPE, rvMsgList);
        }
        SkinHelper.updateBackground(SkinHelper.SKIN_INPUT_AREA, etInput);
        SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BOTTOM_AREA, bottomLayout);
        SkinHelper.updateIcon(SkinHelper.SKIN_SEND_MSG, btnSend);

        etInput.addTextChangedListener(new TextWatcherAdapter() {
            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                onTextChange(s);
                updateSendButtonStatus(s);
            }
        });

        helper = new KeyboardChangeHelper(getView());
        helper.addListener();
        helper.addOnKeyboardShowListener(new KeyboardChangeHelper.OnKeyboardShowListener() {
            @Override
            public void onKeyboardShow() {
                if (!linearLayoutManager.getStackFromEnd()) {
                    int firstVisibleItemPosition = linearLayoutManager.findFirstCompletelyVisibleItemPosition();
                    if (firstVisibleItemPosition > 0) linearLayoutManager.setStackFromEnd(true);
                }
                scrollRecyclerViewToEnd();
            }

            @Override
            public void onKeyboardHide() {
            }
        });

        btnSend.setOnClickListener(this);
        btnSend.setClickable(false);
        btnSend.setEnabled(false);

    }

    private void updateSendButtonStatus(CharSequence s) {
        VectorDrawableCompat drawable = VectorDrawableCompat.create(getResources(), R.drawable.aihelp_svg_ic_send_msg, null);
        if (drawable != null) {
            int greyColor = Color.parseColor("#C6C9D7");
            int blueColor = Color.parseColor("#5C7EFF");
            if (!TextUtils.isEmpty(s.toString().trim())) {
                btnSend.setClickable(true);
                DrawableCompat.setTint(DrawableCompat.wrap(drawable).mutate(), blueColor);
            } else {
                btnSend.setClickable(false);
                DrawableCompat.setTint(DrawableCompat.wrap(drawable).mutate(), greyColor);
            }
            if (TextUtils.isEmpty(SkinHelper.getSkinResource(SkinHelper.SKIN_SEND_MSG))) {
                btnSend.setImageDrawable(drawable);
            }
        }
    }

    protected void onTextChange(CharSequence s) {

    }

    public void updateChatList(ConversationMsg msg) {

        if (Const.TOGGLE_CHAT_TIMESTAMP && msg.getMsgType() != ConversationMsg.TYPE_TIMESTAMP) {
            prepareTimeStamp(msg.getTimeStamp());
        }

        if (msg.getMsgType() == ConversationMsg.TYPE_USER_TEXT ||
                msg.getMsgType() == ConversationMsg.TYPE_USER_TEXT_BOT) {
            etInput.setText("");
            updateSendButtonStatus("");
        }
        mAdapter.update(msg);
        scrollRecyclerViewToEnd();
    }

    private void scrollRecyclerViewToEnd() {
        rvMsgList.post(new Runnable() {
            @Override
            public void run() {
                rvMsgList.scrollToPosition(mAdapter.getItemCount() - 1);
            }
        });
    }

    public void updateChatListClearly(ConversationMsg msg) {
        ArrayList<ConversationMsg> conversationList = new ArrayList<>();
        conversationList.add(msg);
        updateChatList(conversationList);
    }

    public void updateChatList(List<ConversationMsg> msg) {
        mAdapter.update(msg, true);
        scrollRecyclerViewToEnd();
    }

    @Override
    public void onConfigurationChanged(@NonNull Configuration newConfig) {
        super.onConfigurationChanged(newConfig);
        SoftInputUtil.hideSoftInput(getContext(), rvMsgList);
        rvMsgList.setBackgroundColor(Color.parseColor("#EFF1F6"));
        if (getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT) {
            if (!TextUtils.isEmpty(SkinHelper.getSkinResource(SkinHelper.SKIN_CHAT_BACKGROUND_PORTRAIT))) {
                SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BACKGROUND_PORTRAIT, rvMsgList);
            }
        } else {
            if (!TextUtils.isEmpty(SkinHelper.getSkinResource(SkinHelper.SKIN_CHAT_BACKGROUND_LANDSCAPE))) {
                SkinHelper.updateBackground(SkinHelper.SKIN_CHAT_BACKGROUND_LANDSCAPE, rvMsgList);
            }
        }
    }

    public void onMqttLogin(List<ConversationMsg> mqttReplyMsg) {
        btnSend.setEnabled(true);
        // Must ensure current page is visible, or the page status will be messed up.
        if (isVisible()) {
            if (this instanceof ElvaBotFragment && getArguments() != null) {
                isAlwaysShowSupportInElva = getArguments().getBoolean(IntentValues.CONTACT_US_ALWAYS_ONLINE);
                if (isAlwaysShowSupportInElva) {
                    EventBus.getDefault().post(new SupportActionEvent(IntentValues.SHOW_SUPPORT_ACTION,
                            IntentValues.SUPPORT_ACTION_MSG));
                }
                if (!TextUtils.isEmpty(Const.CUSTOM_STORY_NODE) || !Const.isLocalWelcomeAvailable) {
                    updateChatList(mqttReplyMsg);
                }
            }

            if (this instanceof ConversationFragment) {
                updateChatList(mqttReplyMsg);
            }
        }
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        if (helper != null) {
            helper.removeListener();
        }
    }

    public void updateNetCheckingStatus(boolean startChecking) {

    }

    public boolean isNetCheckingInProgress() {
        return false;
    }

    @Override
    protected boolean isBindEventBus() {
        return true;
    }

    private void prepareTimeStamp(long timeStamp) {
        if (mTimeStampMsg == null || timeStamp - Long.parseLong(mTimeStampMsg.getMsgContent()) > 5 * 60 * 1000) {
            mTimeStampMsg = new ConversationMsg(ConversationMsg.TYPE_TIMESTAMP, ConversationMsg.STATUS_SUCCESS);
            mTimeStampMsg.setMsgContent(timeStamp + "");
            updateChatList(mTimeStampMsg);
        }
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onEventComing(PrepareMessageTimeStampEvent event) {
        if (Const.TOGGLE_CHAT_TIMESTAMP) {
            ConversationMsg newMsg = (ConversationMsg) event.getEvent();
            prepareTimeStamp(newMsg.getTimeStamp());
        }
    }

}
